<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

session_start();
date_default_timezone_set('America/Sao_Paulo');

// Verificar login e permissões de médico
if (!isLoggedIn() || $_SESSION['user']['type'] !== 'doctor') {
    header("Location: ../index.php");
    exit();
}

$db = Database::getInstance()->getConnection();

// Buscar dados do médico
$doctor_id = $_SESSION['user']['id'];
$stmt = $db->prepare("
    SELECT d.*, s.name as specialty_name, o.name as office_name
    FROM doctors d
    LEFT JOIN specialties s ON s.id = d.specialty_id
    LEFT JOIN offices o ON o.id = d.office_id
    WHERE d.id = :id
");
$stmt->execute([':id' => $doctor_id]);
$doctor = $stmt->fetch();

// Buscar pacientes aguardando para este médico
$stmt = $db->prepare("
    SELECT p.*, pr.name as procedure_name
    FROM patients p
    LEFT JOIN procedures pr ON pr.id = p.procedure_id
    WHERE p.office_id = :office_id
    AND p.called = 0
    ORDER BY 
        CASE p.priority 
            WHEN 'Alta' THEN 1 
            WHEN 'Media' THEN 2 
            ELSE 3 
        END,
        p.arrival_time ASC
");
$stmt->execute([':office_id' => $doctor['office_id']]);
$waiting_patients = $stmt->fetchAll();

// Buscar pacientes já chamados hoje
$stmt = $db->prepare("
    SELECT p.*, pr.name as procedure_name
    FROM patients p
    LEFT JOIN procedures pr ON pr.id = p.procedure_id
    WHERE p.office_id = :office_id
    AND p.called = 1
    AND DATE(p.created_at) = CURDATE()
    ORDER BY p.called_time DESC
");
$stmt->execute([':office_id' => $doctor['office_id']]);
$called_patients = $stmt->fetchAll();

// Processar chamada de paciente
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['call_patient'])) {
    $patient_id = $_POST['call_patient'];
    if (callPatient($patient_id)) {
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel do Médico</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8">
        <div class="flex justify-between items-center mb-8">
            <div>
                <h1 class="text-3xl font-bold mb-2"><?= htmlspecialchars($doctor['name']) ?></h1>
                <p class="text-gray-600">
                    <?= htmlspecialchars($doctor['specialty_name']) ?> - 
                    <?= htmlspecialchars($doctor['office_name']) ?>
                </p>
            </div>
            <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-4 rounded">
                Sair
            </a>
        </div>

        <!-- Pacientes Aguardando -->
        <div class="bg-white shadow rounded-lg p-6 mb-6">
            <h2 class="text-2xl font-bold mb-4">Pacientes Aguardando</h2>
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="bg-gray-100">
                            <th class="px-4 py-2 text-left">Nome</th>
                            <th class="px-4 py-2 text-left">Chegada</th>
                            <th class="px-4 py-2 text-left">Tipo</th>
                            <th class="px-4 py-2 text-left">Procedimento</th>
                            <th class="px-4 py-2 text-left">Prioridade</th>
                            <th class="px-4 py-2 text-center">Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($waiting_patients as $patient): ?>
                            <tr class="border-t">
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['name']) ?></td>
                                <td class="px-4 py-2"><?= $patient['arrival_time'] ?></td>
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['attendance_type']) ?></td>
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['procedure_name'] ?? '-') ?></td>
                                <td class="px-4 py-2">
                                    <span class="<?= getPriorityClass($patient['priority']) ?>">
                                        <?= $patient['priority'] ?>
                                    </span>
                                </td>
                                <td class="px-4 py-2 text-center">
                                    <form method="post" class="inline">
                                        <input type="hidden" name="call_patient" value="<?= $patient['id'] ?>">
                                        <button type="submit" 
                                                class="bg-green-500 hover:bg-green-600 text-white font-bold py-1 px-3 rounded">
                                            Chamar
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if (empty($waiting_patients)): ?>
                            <tr>
                                <td colspan="6" class="text-center py-4 text-gray-500">
                                    Nenhum paciente aguardando
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Pacientes Chamados Hoje -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-2xl font-bold mb-4">Atendimentos de Hoje</h2>
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="bg-gray-100">
                            <th class="px-4 py-2 text-left">Nome</th>
                            <th class="px-4 py-2 text-left">Chegada</th>
                            <th class="px-4 py-2 text-left">Chamada</th>
                            <th class="px-4 py-2 text-left">Espera</th>
                            <th class="px-4 py-2 text-left">Tipo</th>
                            <th class="px-4 py-2 text-left">Procedimento</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($called_patients as $patient): ?>
                            <tr class="border-t">
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['name']) ?></td>
                                <td class="px-4 py-2"><?= $patient['arrival_time'] ?></td>
                                <td class="px-4 py-2"><?= $patient['called_time'] ?></td>
                                <td class="px-4 py-2"><?= $patient['waiting_time'] ?></td>
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['attendance_type']) ?></td>
                                <td class="px-4 py-2"><?= htmlspecialchars($patient['procedure_name'] ?? '-') ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if (empty($called_patients)): ?>
                            <tr>
                                <td colspan="6" class="text-center py-4 text-gray-500">
                                    Nenhum paciente chamado hoje
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        // Atualizar a página a cada 30 segundos
        setInterval(function() {
            window.location.reload();
        }, 30000);
    </script>
</body>
</html>

<?php
function getPriorityClass($priority) {
    switch ($priority) {
        case 'Alta':
            return 'text-red-600 font-bold';
        case 'Media':
            return 'text-yellow-600';
        default:
            return 'text-green-600';
    }
}
?>