<?php
require_once 'database.php';

function authenticateUser($username, $password) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("SELECT * FROM users WHERE username = :username");
    $stmt->execute([':username' => $username]);
    $user = $stmt->fetch();
    
    if ($user && password_verify($password, $user['password'])) {
        return [
            'id' => $user['id'],
            'username' => $user['username'],
            'name' => $user['name'],
            'type' => $user['type'],
            'office_id' => $user['office_id']
        ];
    }
    
    return false;
}

function isLoggedIn() {
    return isset($_SESSION['user']);
}

function requireLogin() {
    if (!isLoggedIn()) {
        header("Location: ../index.php");
        exit();
    }
}

function redirectByUserType($user) {
    if (!isset($user['type'])) {
        header("Location: index.php");
        exit();
    }

    switch ($user['type']) {
        case 'admin':
            header("Location: admin/index.php");
            break;
        case 'doctor':
            header("Location: medico/index.php");
            break;
        default:
            header("Location: index.php");
    }
    exit();
}

// Novas funções úteis para o banco de dados
function getSpecialties() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM specialties ORDER BY name");
    return $stmt->fetchAll();
}

function getOffices() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM offices ORDER BY name");
    return $stmt->fetchAll();
}

function getProcedures() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM procedures ORDER BY name");
    return $stmt->fetchAll();
}

function getPaymentMethods() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM payment_methods ORDER BY name");
    return $stmt->fetchAll();
}

function getDoctorById($id) {
    $db = Database::getInstance()->getConnection();
    
    // Buscar dados do médico
    $stmt = $db->prepare("
        SELECT d.*, u.username 
        FROM doctors d 
        LEFT JOIN users u ON u.id = d.id 
        WHERE d.id = :id
    ");
    $stmt->execute([':id' => $id]);
    return $stmt->fetch();
}

function saveDoctor($doctorData, $userData) {
    $db = Database::getInstance()->getConnection();
    
    try {
        $db->beginTransaction();
        
        // Gerar ID único se for novo registro
        $id = $doctorData['id'] ?? uuid();
        
        // Salvar usuário
        if (isset($userData['id'])) {
            // Update
            $stmt = $db->prepare("
                UPDATE users 
                SET username = :username, 
                    name = :name, 
                    office_id = :office_id
                WHERE id = :id
            ");
            $params = [
                ':id' => $id,
                ':username' => $userData['username'],
                ':name' => $userData['name'],
                ':office_id' => $userData['office_id']
            ];
            
            // Se senha foi fornecida, atualizar
            if (!empty($userData['password'])) {
                $stmt = $db->prepare("
                    UPDATE users 
                    SET username = :username, 
                        password = :password,
                        name = :name, 
                        office_id = :office_id
                    WHERE id = :id
                ");
                $params[':password'] = password_hash($userData['password'], PASSWORD_DEFAULT);
            }
            
            $stmt->execute($params);
        } else {
            // Insert
            $stmt = $db->prepare("
                INSERT INTO users (id, username, password, name, type, office_id)
                VALUES (:id, :username, :password, :name, 'doctor', :office_id)
            ");
            $stmt->execute([
                ':id' => $id,
                ':username' => $userData['username'],
                ':password' => password_hash($userData['password'], PASSWORD_DEFAULT),
                ':name' => $userData['name'],
                ':office_id' => $userData['office_id']
            ]);
        }
        
        // Salvar médico
        if (isset($doctorData['id'])) {
            // Update
            $stmt = $db->prepare("
                UPDATE doctors 
                SET name = :name,
                    specialty_id = :specialty_id,
                    whatsapp = :whatsapp,
                    email = :email,
                    office_id = :office_id,
                    consultation_value = :consultation_value,
                    consultation_commission = :consultation_commission
                WHERE id = :id
            ");
        } else {
            // Insert
            $stmt = $db->prepare("
                INSERT INTO doctors (id, name, specialty_id, whatsapp, email, office_id, 
                                   consultation_value, consultation_commission)
                VALUES (:id, :name, :specialty_id, :whatsapp, :email, :office_id, 
                        :consultation_value, :consultation_commission)
            ");
        }
        
        $stmt->execute([
            ':id' => $id,
            ':name' => $doctorData['name'],
            ':specialty_id' => $doctorData['specialty_id'],
            ':whatsapp' => preg_replace('/[^0-9]/', '', $doctorData['whatsapp']),
            ':email' => $doctorData['email'],
            ':office_id' => $doctorData['office_id'],
            ':consultation_value' => str_replace(['R$', '.', ','], ['', '', '.'], $doctorData['consultation_value']),
            ':consultation_commission' => str_replace(['R$', '.', ','], ['', '', '.'], $doctorData['consultation_commission'])
        ]);
        
        // Salvar procedimentos do médico
        if (isset($doctorData['procedures'])) {
            // Remover procedimentos antigos
            $stmt = $db->prepare("DELETE FROM doctor_procedures WHERE doctor_id = :doctor_id");
            $stmt->execute([':doctor_id' => $id]);
            
            // Inserir novos procedimentos
            $stmt = $db->prepare("
                INSERT INTO doctor_procedures (doctor_id, procedure_id) 
                VALUES (:doctor_id, :procedure_id)
            ");
            
            foreach ($doctorData['procedures'] as $procedure_id) {
                $stmt->execute([
                    ':doctor_id' => $id,
                    ':procedure_id' => $procedure_id
                ]);
            }
        }
        
        $db->commit();
        return true;
        
    } catch (Exception $e) {
        $db->rollBack();
        error_log($e->getMessage());
        return false;
    }
}

function getAllDoctors() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("
        SELECT d.*, 
               s.name as specialty_name,
               GROUP_CONCAT(p.name) as procedure_names,
               GROUP_CONCAT(p.id) as procedure_ids
        FROM doctors d
        LEFT JOIN specialties s ON s.id = d.specialty_id
        LEFT JOIN doctor_procedures dp ON dp.doctor_id = d.id
        LEFT JOIN procedures p ON p.id = dp.procedure_id
        GROUP BY d.id
        ORDER BY d.name
    ");
    return $stmt->fetchAll();
}

// Função helper para gerar UUID
function uuid() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

function getWaitingPatients($office_id = null) {
    $db = Database::getInstance()->getConnection();
    
    $sql = "
        SELECT p.*, o.name as office_name, pr.name as procedure_name
        FROM patients p
        LEFT JOIN offices o ON o.id = p.office_id
        LEFT JOIN procedures pr ON pr.id = p.procedure_id
        WHERE p.called = 0
    ";
    
    if ($office_id && $office_id !== 'all') {
        $sql .= " AND p.office_id = :office_id";
    }
    
    $sql .= " ORDER BY 
        CASE p.priority 
            WHEN 'Alta' THEN 1 
            WHEN 'Media' THEN 2 
            ELSE 3 
        END,
        p.arrival_time ASC";
    
    $stmt = $office_id && $office_id !== 'all' 
        ? $db->prepare($sql) 
        : $db->query($sql);
    
    if ($office_id && $office_id !== 'all') {
        $stmt->execute([':office_id' => $office_id]);
    }
    
    return $stmt->fetchAll();
}

function getCalledPatients($office_id = null) {
    $db = Database::getInstance()->getConnection();
    
    $sql = "
        SELECT p.*, o.name as office_name, pr.name as procedure_name
        FROM patients p
        LEFT JOIN offices o ON o.id = p.office_id
        LEFT JOIN procedures pr ON pr.id = p.procedure_id
        WHERE p.called = 1
        AND DATE(p.created_at) = CURDATE()
    ";
    
    if ($office_id && $office_id !== 'all') {
        $sql .= " AND p.office_id = :office_id";
    }
    
    $sql .= " ORDER BY p.called_time DESC";
    
    $stmt = $office_id && $office_id !== 'all' 
        ? $db->prepare($sql) 
        : $db->query($sql);
    
    if ($office_id && $office_id !== 'all') {
        $stmt->execute([':office_id' => $office_id]);
    }
    
    return $stmt->fetchAll();
}

function addPatient($data) {
    $db = Database::getInstance()->getConnection();
    
    $data['id'] = uniqid();
    $data['arrival_time'] = date('H:i:s');
    $data['called'] = 0;
    $data['displayed'] = 0;
    
    try {
        return $db->insert('patients', $data);
    } catch (Exception $e) {
        error_log($e->getMessage());
        return false;
    }
}

function callPatient($id) {
    $db = Database::getInstance()->getConnection();
    
    try {
        $stmt = $db->prepare("SELECT arrival_time FROM patients WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $patient = $stmt->fetch();
        
        if (!$patient) return false;
        
        return $db->update('patients', 
            [
                'called' => 1,
                'called_time' => date('H:i:s'),
                'waiting_time' => calculateWaitingTime($patient['arrival_time'], date('H:i:s'))
            ],
            ['id' => $id]
        );
    } catch (Exception $e) {
        error_log($e->getMessage());
        return false;
    }
}

function deletePatient($id) {
    $db = Database::getInstance()->getConnection();
    
    try {
        return $db->delete('patients', ['id' => $id]);
    } catch (Exception $e) {
        error_log($e->getMessage());
        return false;
    }
}

function clearAllPatients() {
    $db = Database::getInstance()->getConnection();
    
    try {
        return $db->query("DELETE FROM patients WHERE called = 1");
    } catch (Exception $e) {
        error_log($e->getMessage());
        return false;
    }
}

function calculateWaitingTime($arrival_time, $called_time) {
    $arrival = strtotime($arrival_time);
    $called = strtotime($called_time);
    $diff = $called - $arrival;
    
    $hours = floor($diff / 3600);
    $mins = floor(($diff - ($hours * 3600)) / 60);
    $secs = $diff - ($hours * 3600) - ($mins * 60);
    
    return sprintf("%02d:%02d:%02d", $hours, $mins, $secs);
}

function cleanMonetaryValue($value) {
    return str_replace(
        ['R$', ' ', '.', ','], 
        ['', '', '', '.'], 
        $value
    );
}

function getPriorityClass($priority) {
    switch ($priority) {
        case 'Alta':
            return 'text-red-600 font-bold';
        case 'Media':
            return 'text-yellow-600';
        default:
            return 'text-green-600';
    }
}