<?php
require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../includes/auth.php';

header('Content-Type: application/json');

// Verifica se é admin
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    die(json_encode(['success' => false, 'message' => 'Acesso negado']));
}

$database = new Database();
$db = $database->getConnection();

// Função para processar upload da foto
function processarFoto($foto) {
    $upload_dir = '../../uploads/medicos/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0777, true);
    }

    $filename = uniqid() . '_' . $foto['name'];
    $upload_file = $upload_dir . $filename;

    if (move_uploaded_file($foto['tmp_name'], $upload_file)) {
        return $filename;
    }
    return false;
}

try {
    // READ - Buscar médico
    if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'read') {
        $query = "SELECT m.*, u.nome, u.login 
                 FROM medicos m 
                 JOIN usuarios u ON m.usuario_id = u.id 
                 WHERE m.id = :id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_GET['id']);
        $stmt->execute();
        
        $medico = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($medico) {
            echo json_encode($medico);
        } else {
            echo json_encode(['success' => false, 'message' => 'Médico não encontrado']);
        }
        exit;
    }

    // CREATE e UPDATE
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        if ($_POST['action'] === 'create') {
            $db->beginTransaction(); // Inicia transação
            
            try {
                // Primeiro criar o usuário com tipo 'medico'
                $query = "INSERT INTO usuarios (nome, login, senha, tipo) 
                         VALUES (:nome, :login, :senha, 'medico')";
                
                $stmt = $db->prepare($query);
                $senha_hash = password_hash($_POST['senha'], PASSWORD_DEFAULT);
                
                $stmt->bindParam(':nome', $_POST['nome']);
                $stmt->bindParam(':login', $_POST['login']);
                $stmt->bindParam(':senha', $senha_hash);
                
                if (!$stmt->execute()) {
                    throw new Exception("Erro ao criar usuário: " . implode(", ", $stmt->errorInfo()));
                }
                
                $usuario_id = $db->lastInsertId();
                
                // Processar foto se existir
                $foto_nome = null;
                if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
                    $foto = $_FILES['foto'];
                    $foto_nome = time() . '_' . $foto['name'];
                    $foto_destino = '../../uploads/medicos/' . $foto_nome;
                    
                    if (!move_uploaded_file($foto['tmp_name'], $foto_destino)) {
                        throw new Exception("Erro ao salvar foto");
                    }
                }
                
                // Tratar campos vazios
                $crm = !empty($_POST['crm']) ? $_POST['crm'] : '';
                $whatsapp = !empty($_POST['whatsapp']) ? $_POST['whatsapp'] : null;
                $observacoes = !empty($_POST['observacoes']) ? $_POST['observacoes'] : null;
                
                // Agora criar o médico
                $query = "INSERT INTO medicos (usuario_id, crm, especialidade_id, consultorio_id, 
                                             whatsapp, status, foto, observacoes) 
                         VALUES (:usuario_id, :crm, :especialidade_id, :consultorio_id,
                                 :whatsapp, :status, :foto, :observacoes)";
                
                $stmt = $db->prepare($query);
                
                $stmt->bindParam(':usuario_id', $usuario_id);
                $stmt->bindParam(':crm', $crm);
                $stmt->bindParam(':especialidade_id', $_POST['especialidade_id']);
                $stmt->bindParam(':consultorio_id', $_POST['consultorio_id']);
                $stmt->bindParam(':whatsapp', $whatsapp);
                $stmt->bindParam(':status', $_POST['status']);
                $stmt->bindParam(':foto', $foto_nome);
                $stmt->bindParam(':observacoes', $observacoes);
                
                if (!$stmt->execute()) {
                    throw new Exception("Erro ao criar médico: " . implode(", ", $stmt->errorInfo()));
                }
                
                $db->commit(); // Confirma transação
                echo json_encode(['success' => true, 'message' => 'Médico cadastrado com sucesso']);
                
            } catch (Exception $e) {
                $db->rollBack(); // Desfaz transação em caso de erro
                throw $e;
            }
        } elseif ($_POST['action'] === 'update') {
            $db->beginTransaction();

            // Processa nova foto se foi enviada
            if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
                $foto = processarFoto($_FILES['foto']);
                if ($foto) {
                    // Atualiza a foto no banco
                    $query = "UPDATE medicos SET foto = :foto WHERE id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->bindParam(':foto', $foto);
                    $stmt->bindParam(':id', $_POST['id']);
                    $stmt->execute();
                }
            }

            // 1. Atualiza dados do médico
            $query = "UPDATE medicos SET 
                     especialidade_id = :especialidade_id,
                     crm = :crm,
                     whatsapp = :whatsapp,
                     consultorio_id = :consultorio_id,
                     status = :status
                     WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_POST['id']);
            $stmt->bindParam(':especialidade_id', $_POST['especialidade_id']);
            $stmt->bindParam(':crm', $_POST['crm']);
            $stmt->bindParam(':whatsapp', $_POST['whatsapp']);
            $stmt->bindParam(':consultorio_id', $_POST['consultorio_id']);
            $stmt->bindParam(':status', $_POST['status']);
            $stmt->execute();

            // 2. Atualiza dados do usuário
            $query = "UPDATE usuarios SET 
                     nome = :nome,
                     login = :login
                     WHERE id = (SELECT usuario_id FROM medicos WHERE id = :id)";
                     
            $stmt = $db->prepare($query);
            $stmt->bindParam(':nome', $_POST['nome']);
            $stmt->bindParam(':login', $_POST['login']);
            $stmt->bindParam(':id', $_POST['id']);
            $stmt->execute();

            // 3. Atualiza senha se fornecida
            if (!empty($_POST['senha'])) {
                $query = "UPDATE usuarios SET senha = :senha 
                         WHERE id = (SELECT usuario_id FROM medicos WHERE id = :id)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':senha', $_POST['senha']);
                $stmt->bindParam(':id', $_POST['id']);
                $stmt->execute();
            }

            $db->commit();
            echo json_encode(['success' => true, 'message' => 'Médico atualizado com sucesso']);

        } elseif ($_POST['action'] === 'delete') {
            $db->beginTransaction();

            $id = $_POST['id'];

            // Pega o usuario_id antes de deletar
            $query = "SELECT usuario_id FROM medicos WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $usuario_id = $stmt->fetch(PDO::FETCH_ASSOC)['usuario_id'];

            // Deleta o médico
            $query = "DELETE FROM medicos WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            // Deleta o usuário
            $query = "DELETE FROM usuarios WHERE id = :usuario_id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':usuario_id', $usuario_id);
            $stmt->execute();

            $db->commit();
            echo json_encode(['success' => true, 'message' => 'Médico excluído com sucesso']);

        } else {
            throw new Exception("Ação inválida");
        }
    } else {
        throw new Exception("Método ou ação não especificada");
    }
    
} catch (Exception $e) {
    error_log("Erro: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} 