<?php
require_once '../../config/config.php';
require_once '../../config/database.php';

header('Content-Type: application/json');

// Verifica se é admin ou médico
if (!isset($_SESSION['user_type']) || !in_array($_SESSION['user_type'], ['admin', 'medico'])) {
    die(json_encode(['success' => false, 'message' => 'Acesso negado']));
}

$database = new Database();
$db = $database->getConnection();

try {
    // Buscar paciente
    if (isset($_GET['action']) && $_GET['action'] === 'buscar_paciente') {
        try {
            $busca = $_GET['busca'];
            
            // Remove caracteres especiais se for CPF
            $buscaSemMascara = preg_replace('/[^0-9]/', '', $busca);
            
            // Se for número com 11 dígitos, considera CPF
            if (strlen($buscaSemMascara) === 11) {
                // Formata o CPF
                $cpfFormatado = substr($buscaSemMascara, 0, 3) . '.' . 
                               substr($buscaSemMascara, 3, 3) . '.' . 
                               substr($buscaSemMascara, 6, 3) . '-' . 
                               substr($buscaSemMascara, 9, 2);
                
                $query = "SELECT id, nome, cpf FROM pacientes 
                         WHERE cpf = :cpf 
                         AND status = 'ativo'
                         LIMIT 1";
                
                $stmt = $db->prepare($query);
                $stmt->bindParam(':cpf', $cpfFormatado);
            } else {
                // Busca por nome
                $query = "SELECT id, nome, cpf FROM pacientes 
                         WHERE nome LIKE :nome 
                         AND status = 'ativo'
                         ORDER BY nome
                         LIMIT 10";
                
                $stmt = $db->prepare($query);
                $stmt->bindValue(':nome', "%$busca%");
            }
            
            $stmt->execute();
            $pacientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($pacientes) > 0) {
                if (count($pacientes) === 1) {
                    echo json_encode(['success' => true, 'paciente' => $pacientes[0]]);
                } else {
                    echo json_encode(['success' => true, 'multiplos' => true, 'pacientes' => $pacientes]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Paciente não encontrado']);
            }
            
        } catch(PDOException $e) {
            error_log("Erro ao buscar paciente: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erro ao buscar paciente']);
        }
        exit;
    }

    // Adicionar à fila
    if (isset($_POST['action']) && $_POST['action'] === 'adicionar_fila') {
        try {
            // Verifica se paciente já está na fila deste médico
            $query = "SELECT id FROM fila_atendimento 
                     WHERE paciente_id = :paciente_id 
                     AND medico_id = :medico_id 
                     AND status IN ('aguardando', 'chamado')";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $_POST['paciente_id']);
            $stmt->bindParam(':medico_id', $_POST['medico_id']);
            $stmt->execute();
            
            if ($stmt->fetch()) {
                echo json_encode(['success' => false, 'message' => 'Paciente já está na fila deste médico']);
                exit;
            }

            // Ajusta o tipo de atendimento para ser apenas 'chegada' ou 'agendado'
            $tipo_atendimento = $_POST['tipo_atendimento'] === 'agendado' ? 'agendado' : 'chegada';

            $query = "INSERT INTO fila_atendimento (
                paciente_id, medico_id, consultorio_id, tipo_atendimento,
                horario_agendado, prioridade, observacoes, status
            ) VALUES (
                :paciente_id, :medico_id, :consultorio_id, :tipo_atendimento,
                :horario_agendado, :prioridade, :observacoes, 'aguardando'
            )";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $_POST['paciente_id']);
            $stmt->bindParam(':medico_id', $_POST['medico_id']);
            $stmt->bindParam(':consultorio_id', $_POST['consultorio_id']);
            $stmt->bindParam(':tipo_atendimento', $tipo_atendimento);
            $stmt->bindParam(':horario_agendado', $_POST['horario_agendado']);
            $stmt->bindParam(':prioridade', $_POST['prioridade']);
            $stmt->bindParam(':observacoes', $_POST['observacoes']);
            
            $stmt->execute();
            echo json_encode(['success' => true, 'message' => 'Paciente adicionado à fila com sucesso']);

        } catch(Exception $e) {
            error_log("Erro ao adicionar à fila: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erro ao adicionar paciente à fila: ' . $e->getMessage()]);
        }
        exit;
    }

    // Chamar paciente
    if (isset($_POST['action']) && $_POST['action'] === 'chamar') {
        try {
            // Se for médico, verifica se o paciente pertence a ele
            if ($_SESSION['user_type'] === 'medico') {
                $query = "SELECT f.* FROM fila_atendimento f 
                         JOIN medicos m ON f.medico_id = m.id 
                         WHERE f.id = :id AND m.usuario_id = :usuario_id";
                
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id', $_POST['id']);
                $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
                $stmt->execute();
                
                if (!$stmt->fetch()) {
                    throw new Exception('Paciente não encontrado na sua fila');
                }
            }

            // Atualiza o status e horário da chamada
            $query = "UPDATE fila_atendimento 
                     SET status = 'chamado', 
                         called_at = NOW(),
                         chamada_realizada = 0
                     WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_POST['id']);
            $result = $stmt->execute();
            
            if ($result) {
                // Busca os dados da chamada
                $query = "SELECT 
                            f.id,
                            p.nome as paciente_nome,
                            c.descricao as consultorio_descricao,
                            u.nome as medico_nome,
                            m.foto as medico_foto,
                            f.called_at
                         FROM fila_atendimento f
                         JOIN pacientes p ON f.paciente_id = p.id
                         JOIN medicos m ON f.medico_id = m.id
                         JOIN usuarios u ON m.usuario_id = u.id
                         JOIN consultorios c ON f.consultorio_id = c.id
                         WHERE f.id = :id";
                
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id', $_POST['id']);
                $stmt->execute();
                $chamada = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Marca que houve uma nova chamada (mantém a sessão)
                $_SESSION['nova_chamada'] = true;
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Paciente chamado com sucesso',
                    'chamada' => $chamada
                ]);
            } else {
                throw new Exception('Erro ao chamar paciente');
            }
            
        } catch(Exception $e) {
            error_log("Erro ao chamar paciente: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }

    // Remover da fila
    if (isset($_POST['action']) && $_POST['action'] === 'remover') {
        try {
            $query = "UPDATE fila_atendimento 
                     SET status = 'cancelado' 
                     WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_POST['id']);
            $stmt->execute();
            
            echo json_encode(['success' => true, 'message' => 'Paciente removido da fila com sucesso']);
            
        } catch(PDOException $e) {
            error_log("Erro ao remover paciente: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erro ao remover paciente da fila']);
        }
        exit;
    }

    // Buscar paciente da fila
    if (isset($_GET['action']) && $_GET['action'] === 'buscar' && isset($_GET['id'])) {
        try {
            // Alterando a query para buscar todos os campos necessários
            $query = "SELECT 
                        f.id,
                        f.tipo_atendimento,
                        f.horario_agendado,
                        f.prioridade,
                        f.observacoes,
                        f.status,
                        p.nome as paciente_nome,
                        p.cpf,
                        u.nome as medico_nome,
                        c.descricao as consultorio_nome
                    FROM fila_atendimento f
                    JOIN pacientes p ON f.paciente_id = p.id
                    JOIN medicos m ON f.medico_id = m.id
                    JOIN usuarios u ON m.usuario_id = u.id
                    JOIN consultorios c ON f.consultorio_id = c.id
                    WHERE f.id = :id";

            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_GET['id']);
            $stmt->execute();
            
            $paciente = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($paciente) {
                // Debug
                error_log("Dados do paciente encontrados: " . print_r($paciente, true));
                echo json_encode(['success' => true, 'paciente' => $paciente]);
            } else {
                error_log("Paciente não encontrado na fila. ID: " . $_GET['id']);
                echo json_encode(['success' => false, 'message' => 'Paciente não encontrado na fila']);
            }
            
        } catch(PDOException $e) {
            error_log("Erro ao buscar paciente da fila: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erro ao buscar paciente']);
        }
        exit;
    }

    // Editar paciente na fila
    if (isset($_POST['action']) && $_POST['action'] === 'editar') {
        try {
            error_log("Dados recebidos: " . print_r($_POST, true)); // Debug

            // Normaliza o texto da prioridade
            $prioridade = strtolower($_POST['prioridade']);
            switch ($prioridade) {
                case 'pcd':
                    $prioridade = 'Pcd';
                    break;
                case 'normal':
                    $prioridade = 'Normal';
                    break;
                case 'urgencia':
                case 'urgência':
                    $prioridade = 'Urgencia';
                    break;
                case 'idoso':
                    $prioridade = 'Idoso';
                    break;
                case 'gestante':
                    $prioridade = 'Gestante';
                    break;
                default:
                    throw new Exception('Prioridade inválida');
            }

            // Query para atualizar todos os campos
            $query = "UPDATE fila_atendimento 
                     SET prioridade = :prioridade,
                         tipo_atendimento = :tipo_atendimento,
                         horario_agendado = :horario_agendado,
                         observacoes = :observacoes
                     WHERE id = :id";
                
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_POST['id']);
            $stmt->bindParam(':prioridade', $prioridade);
            $stmt->bindParam(':tipo_atendimento', $_POST['tipo_atendimento']);
            $stmt->bindParam(':horario_agendado', $_POST['horario_agendado']);
            $stmt->bindParam(':observacoes', $_POST['observacoes']);
            
            $result = $stmt->execute();
            error_log("Resultado da execução: " . ($result ? "sucesso" : "falha")); // Debug
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Paciente atualizado com sucesso']);
            } else {
                throw new Exception('Erro ao atualizar registro');
            }
            
        } catch(Exception $e) {
            error_log("Erro ao editar paciente na fila: " . $e->getMessage());
            error_log("Stack trace: " . $e->getTraceAsString()); // Debug
            echo json_encode(['success' => false, 'message' => 'Erro ao atualizar paciente: ' . $e->getMessage()]);
        }
        exit;
    }

    // Buscar pacientes da fila
    if (isset($_GET['action']) && $_GET['action'] === 'buscar_fila') {
        try {
            $query = "SELECT 
                f.id, 
                f.tipo_atendimento,
                f.prioridade,
                f.horario_agendado,
                f.status,
                f.created_at,
                f.called_at,
                p.nome as paciente_nome,
                u.nome as medico_nome,
                c.descricao as consultorio_nome
              FROM fila_atendimento f
              JOIN pacientes p ON f.paciente_id = p.id
              JOIN medicos m ON f.medico_id = m.id
              JOIN usuarios u ON m.usuario_id = u.id
              JOIN consultorios c ON f.consultorio_id = c.id
              WHERE f.status IN ('aguardando', 'chamado')";

            // Se for médico, filtra apenas seus pacientes
            if ($_SESSION['user_type'] === 'medico') {
                $query .= " AND m.usuario_id = :usuario_id";
            }
            // Se for admin e tiver filtro de médico
            else if (isset($_GET['medico_nome']) && $_GET['medico_nome'] !== 'Todos os Médicos') {
                $query .= " AND u.nome = :medico_nome";
            }

            $query .= " ORDER BY 
                CASE f.prioridade
                    WHEN 'urgencia' THEN 1
                    WHEN 'idoso' THEN 2
                    WHEN 'pcd' THEN 3
                    WHEN 'gestante' THEN 4
                    WHEN 'normal' THEN 5
                END ASC,
                f.created_at ASC";

            $stmt = $db->prepare($query);
            
            // Bind params conforme o caso
            if ($_SESSION['user_type'] === 'medico') {
                $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
            }
            else if (isset($_GET['medico_nome']) && $_GET['medico_nome'] !== 'Todos os Médicos') {
                $stmt->bindParam(':medico_nome', $_GET['medico_nome']);
            }
            
            $stmt->execute();
            $pacientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'pacientes' => $pacientes]);
        } catch(PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao buscar pacientes da fila']);
        }
        exit;
    }

    // READ - Buscar paciente da fila
    if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'read') {
        $query = "SELECT f.*, p.nome as paciente_nome 
                 FROM fila_atendimento f
                 JOIN pacientes p ON f.paciente_id = p.id
                 WHERE f.id = :id";
                 
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_GET['id']);
        
        if ($stmt->execute()) {
            $paciente = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($paciente) {
                echo json_encode($paciente);
            } else {
                echo json_encode(['success' => false, 'message' => 'Paciente não encontrado']);
            }
        } else {
            throw new Exception("Erro ao buscar paciente");
        }
        exit;
    }

    // Verificar a parte de DELETE
    if (isset($_POST['action']) && $_POST['action'] === 'delete') {
        try {
            // Verifica se o ID foi fornecido
            if (!isset($_POST['id'])) {
                throw new Exception('ID não fornecido');
            }

            $id = $_POST['id'];

            // Verifica se o registro existe antes de deletar
            $query = "SELECT id FROM fila_atendimento WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            if (!$stmt->fetch()) {
                throw new Exception('Registro não encontrado');
            }

            // Deleta o registro
            $query = "DELETE FROM fila_atendimento WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Paciente removido da fila com sucesso']);
            } else {
                throw new Exception('Erro ao remover paciente da fila');
            }

        } catch(Exception $e) {
            error_log("Erro ao excluir da fila: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }

    // Resto do código para outras ações (UPDATE, DELETE, etc)...
} catch (Exception $e) {
    error_log("Erro na fila: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} 