<?php
require_once '../../config/config.php';
require_once '../../includes/auth_medico.php';
require_once '../../includes/header.php';

try {
    $database = new Database();
    $db = $database->getConnection();

    // Buscar dados do médico logado
    $query = "SELECT m.*, u.nome as nome_medico, c.descricao as consultorio_nome
             FROM medicos m 
             JOIN usuarios u ON m.usuario_id = u.id
             LEFT JOIN consultorios c ON m.consultorio_id = c.id
             WHERE m.usuario_id = :usuario_id";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
    $stmt->execute();
    $medico = $stmt->fetch(PDO::FETCH_ASSOC);

    // Buscar fila apenas deste médico (igual ao admin, mas filtrado por médico)
    $query = "SELECT 
                f.id, 
                f.tipo_atendimento, 
                f.prioridade, 
                f.status,
                f.horario_agendado,
                f.created_at,
                f.called_at,
                p.nome as paciente_nome, 
                c.descricao as consultorio_nome
              FROM fila_atendimento f
              JOIN pacientes p ON f.paciente_id = p.id
              JOIN consultorios c ON f.consultorio_id = c.id
              WHERE f.medico_id = :medico_id 
              AND f.status IN ('aguardando', 'chamado')
              ORDER BY 
                CASE f.status
                    WHEN 'chamado' THEN 2
                    ELSE 1
                END,
                CASE 
                    WHEN f.status = 'aguardando' THEN
                        CASE f.prioridade
                            WHEN 'urgencia' THEN 1
                            WHEN 'idoso' THEN 2
                            WHEN 'pcd' THEN 3
                            WHEN 'gestante' THEN 4
                            ELSE 5
                        END
                    ELSE f.called_at
                END,
                f.created_at ASC";

    $stmt = $db->prepare($query);
    $stmt->bindParam(':medico_id', $medico['id']);
    $stmt->execute();
    $fila = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (Exception $e) {
    die("Erro ao carregar a fila. Por favor, contate o administrador.");
}
?>

<div class="container mt-4">
    <div class="card mb-4">
        <div class="card-body">
            <div class="d-flex align-items-center">
                <img src="<?php echo $medico['foto'] ? BASE_URL . '/uploads/medicos/' . $medico['foto'] : 'https://cdn.pixabay.com/photo/2016/08/08/09/17/avatar-1577909_1280.png'; ?>" 
                     class="rounded-circle me-3" 
                     width="60" height="60" 
                     alt="Foto do médico">
                <div>
                    <h4 class="mb-0">Dr(a). <?php echo htmlspecialchars($medico['nome_medico']); ?></h4>
                    <?php if ($medico['consultorio_nome']): ?>
                        <p class="text-muted mb-0">Consultório: <?php echo htmlspecialchars($medico['consultorio_nome']); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Versão Desktop -->
    <div class="d-none d-md-block">
        <ul class="nav nav-tabs mb-4">
            <li class="nav-item">
                <a class="nav-link active" data-bs-toggle="tab" href="#aguardando-desktop">Aguardando</a>
            </li>
            <li class="nav-item">
                <a class="nav-link" data-bs-toggle="tab" href="#chamados-desktop">Chamados</a>
            </li>
        </ul>

        <div class="tab-content">
            <!-- Aba Aguardando Desktop -->
            <div class="tab-pane fade show active" id="aguardando-desktop">
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Paciente</th>
                                <th>Tipo</th>
                                <th>Horário</th>
                                <th>Prioridade</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($fila)): ?>
                                <tr>
                                    <td colspan="5" class="text-center">Nenhum paciente na fila</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($fila as $atendimento): ?>
                                    <?php if ($atendimento['status'] === 'aguardando'): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></td>
                                            <td>
                                                <?php 
                                                    echo $atendimento['tipo_atendimento'] === 'agendado' ? 
                                                        '<span class="badge bg-primary">Agendado</span>' : 
                                                        '<span class="badge bg-secondary">Chegada</span>';
                                                ?>
                                            </td>
                                            <td>
                                                <?php 
                                                    if ($atendimento['tipo_atendimento'] === 'agendado') {
                                                        echo date('H:i', strtotime($atendimento['horario_agendado']));
                                                    } else {
                                                        echo date('H:i', strtotime($atendimento['created_at']));
                                                    }
                                                ?>
                                            </td>
                                            <td>
                                                <?php 
                                                    $badgeClass = '';
                                                    $texto = strtoupper($atendimento['prioridade']);
                                                    
                                                    switch ($atendimento['prioridade']) {
                                                        case 'urgencia':
                                                            $badgeClass = 'bg-danger';
                                                            $texto = 'URGÊNCIA';
                                                            break;
                                                        case 'normal':
                                                            $badgeClass = 'bg-success';
                                                            break;
                                                        default:
                                                            $badgeClass = 'bg-warning';
                                                    }
                                                ?>
                                                <span class="badge <?php echo $badgeClass; ?>"><?php echo $texto; ?></span>
                                            </td>
                                            <td>
                                                <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                                    <i class="fas fa-bullhorn"></i> Chamar
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Aba Chamados Desktop -->
            <div class="tab-pane fade" id="chamados-desktop">
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Paciente</th>
                                <th>Tipo</th>
                                <th>Horário Chamada</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($fila)): ?>
                                <tr>
                                    <td colspan="4" class="text-center">Nenhum paciente chamado</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($fila as $atendimento): ?>
                                    <?php if ($atendimento['status'] === 'chamado'): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></td>
                                            <td>
                                                <?php 
                                                    echo $atendimento['tipo_atendimento'] === 'agendado' ? 
                                                        '<span class="badge bg-primary">Agendado</span>' : 
                                                        '<span class="badge bg-secondary">Chegada</span>';
                                                ?>
                                            </td>
                                            <td><?php echo date('H:i', strtotime($atendimento['called_at'])); ?></td>
                                            <td>
                                                <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                                    <i class="fas fa-bullhorn"></i> Chamar Novamente
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Versão Mobile -->
    <div class="d-md-none">
        <ul class="nav nav-tabs mb-4">
            <li class="nav-item">
                <a class="nav-link active" data-bs-toggle="tab" href="#aguardando-mobile">Aguardando</a>
            </li>
            <li class="nav-item">
                <a class="nav-link" data-bs-toggle="tab" href="#chamados-mobile">Chamados</a>
            </li>
        </ul>

        <div class="tab-content">
            <!-- Aba Aguardando Mobile -->
            <div class="tab-pane fade show active" id="aguardando-mobile">
                <?php if (empty($fila)): ?>
                    <div class="alert alert-info">Nenhum paciente na fila</div>
                <?php else: ?>
                    <?php foreach ($fila as $atendimento): ?>
                        <?php if ($atendimento['status'] === 'aguardando'): ?>
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h5 class="card-title"><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></h5>
                                    
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span class="badge bg-<?php echo $atendimento['tipo_atendimento'] === 'agendado' ? 'primary' : 'secondary'; ?>">
                                            <?php echo ucfirst($atendimento['tipo_atendimento']); ?>
                                        </span>
                                        
                                        <?php 
                                            $badgeClass = '';
                                            $texto = strtoupper($atendimento['prioridade']);
                                            
                                            switch ($atendimento['prioridade']) {
                                                case 'urgencia':
                                                    $badgeClass = 'bg-danger';
                                                    $texto = 'URGÊNCIA';
                                                    break;
                                                case 'normal':
                                                    $badgeClass = 'bg-success';
                                                    break;
                                                default:
                                                    $badgeClass = 'bg-warning';
                                            }
                                        ?>
                                        <span class="badge <?php echo $badgeClass; ?>"><?php echo $texto; ?></span>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            <?php if ($atendimento['tipo_atendimento'] === 'agendado'): ?>
                                                Horário Agendado: <?php echo date('H:i', strtotime($atendimento['horario_agendado'])); ?>
                                            <?php else: ?>
                                                Horário de Chegada: <?php echo date('H:i', strtotime($atendimento['created_at'])); ?>
                                            <?php endif; ?>
                                        </small>
                                    </div>
                                    
                                    <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar
                                    </button>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <!-- Aba Chamados Mobile -->
            <div class="tab-pane fade" id="chamados-mobile">
                <?php if (empty($fila)): ?>
                    <div class="alert alert-info">Nenhum paciente chamado</div>
                <?php else: ?>
                    <?php foreach ($fila as $atendimento): ?>
                        <?php if ($atendimento['status'] === 'chamado'): ?>
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h5 class="card-title"><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></h5>
                                    
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span class="badge bg-<?php echo $atendimento['tipo_atendimento'] === 'agendado' ? 'primary' : 'secondary'; ?>">
                                            <?php echo ucfirst($atendimento['tipo_atendimento']); ?>
                                        </span>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            Chamado às: <?php echo date('H:i', strtotime($atendimento['called_at'])); ?>
                                        </small>
                                    </div>
                                    
                                    <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar Novamente
                                    </button>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
const BASE_URL = '<?php echo BASE_URL; ?>';

function chamarPaciente(id, button) {
    // Adiciona classe de animação
    button.classList.add('btn-chamar-active');
    
    fetch(`${BASE_URL}/src/admin/fila.php`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=chamar&id=${id}`
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Atualiza a fila sem recarregar a página
            atualizarFila();
        } else {
            alert(data.message || 'Erro ao chamar paciente');
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        alert('Erro ao processar requisição');
    })
    .finally(() => {
        // Remove a classe após 1 segundo
        setTimeout(() => {
            button.classList.remove('btn-chamar-active');
        }, 1000);
    });
}

// Variável para controlar se já existe uma atualização em andamento
let atualizacaoEmAndamento = false;

// Função de atualização automática para a página do médico
function atualizarFila() {
    if (atualizacaoEmAndamento) return;
    atualizacaoEmAndamento = true;
    
    // Guarda a aba ativa atual
    const abaAtiva = document.querySelector('.tab-pane.active').id;
    
    fetch(`${BASE_URL}/src/admin/fila.php?action=buscar_fila`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const filaAguardando = data.pacientes.filter(p => p.status === 'aguardando');
                const filaChamados = data.pacientes.filter(p => p.status === 'chamado');
                
                // Atualiza tabela desktop - Aguardando
                const tabelaAguardando = document.querySelector('#aguardando-desktop .table tbody');
                if (filaAguardando.length === 0) {
                    tabelaAguardando.innerHTML = '<tr><td colspan="5" class="text-center">Nenhum paciente na fila</td></tr>';
                } else {
                    tabelaAguardando.innerHTML = filaAguardando.map(atendimento => `
                        <tr>
                            <td>${atendimento.paciente_nome}</td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    '<span class="badge bg-primary">Agendado</span>' : 
                                    '<span class="badge bg-secondary">Chegada</span>'}
                            </td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    new Date(atendimento.horario_agendado).toLocaleTimeString().substr(0,5) : 
                                    new Date(atendimento.created_at).toLocaleTimeString().substr(0,5)}
                            </td>
                            <td>
                                <span class="badge ${getBadgeClass(atendimento.prioridade)}">
                                    ${getPrioridadeTexto(atendimento.prioridade)}
                                </span>
                            </td>
                            <td>
                                <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                    <i class="fas fa-bullhorn"></i> Chamar
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }

                // Atualiza tabela desktop - Chamados
                const tabelaChamados = document.querySelector('#chamados-desktop .table tbody');
                if (filaChamados.length === 0) {
                    tabelaChamados.innerHTML = '<tr><td colspan="4" class="text-center">Nenhum paciente chamado</td></tr>';
                } else {
                    tabelaChamados.innerHTML = filaChamados.map(atendimento => `
                        <tr>
                            <td>${atendimento.paciente_nome}</td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    '<span class="badge bg-primary">Agendado</span>' : 
                                    '<span class="badge bg-secondary">Chegada</span>'}
                            </td>
                            <td>${new Date(atendimento.called_at).toLocaleTimeString().substr(0,5)}</td>
                            <td>
                                <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                    <i class="fas fa-bullhorn"></i> Chamar Novamente
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }

                // Atualiza versão mobile
                const aguardandoMobile = document.querySelector('#aguardando-mobile');
                if (aguardandoMobile) {
                    if (filaAguardando.length === 0) {
                        aguardandoMobile.innerHTML = '<div class="alert alert-info">Nenhum paciente na fila</div>';
                    } else {
                        aguardandoMobile.innerHTML = filaAguardando.map(atendimento => `
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">${atendimento.paciente_nome}</h5>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span class="badge bg-${atendimento.tipo_atendimento === 'agendado' ? 'primary' : 'secondary'}">
                                            ${atendimento.tipo_atendimento === 'agendado' ? 'Agendado' : 'Chegada'}
                                        </span>
                                        <span class="badge ${getBadgeClass(atendimento.prioridade)}">
                                            ${getPrioridadeTexto(atendimento.prioridade)}
                                        </span>
                                    </div>
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            ${atendimento.tipo_atendimento === 'agendado' ? 
                                                'Horário Agendado: ' + new Date(atendimento.horario_agendado).toLocaleTimeString().substr(0,5) : 
                                                'Horário de Chegada: ' + new Date(atendimento.created_at).toLocaleTimeString().substr(0,5)}
                                        </small>
                                    </div>
                                    <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar
                                    </button>
                                </div>
                            </div>
                        `).join('');
                    }
                }

                const chamadosMobile = document.querySelector('#chamados-mobile');
                if (chamadosMobile) {
                    if (filaChamados.length === 0) {
                        chamadosMobile.innerHTML = '<div class="alert alert-info">Nenhum paciente chamado</div>';
                    } else {
                        chamadosMobile.innerHTML = filaChamados.map(atendimento => `
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">${atendimento.paciente_nome}</h5>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span class="badge bg-${atendimento.tipo_atendimento === 'agendado' ? 'primary' : 'secondary'}">
                                            ${atendimento.tipo_atendimento === 'agendado' ? 'Agendado' : 'Chegada'}
                                        </span>
                                    </div>
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            Chamado às: ${new Date(atendimento.called_at).toLocaleTimeString().substr(0,5)}
                                        </small>
                                    </div>
                                    <button class="btn btn-warning btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar Novamente
                                    </button>
                                </div>
                            </div>
                        `).join('');
                    }
                }

                // Mantém a aba ativa
                const tab = new bootstrap.Tab(document.querySelector(`a[href="#${abaAtiva}"]`));
                tab.show();
            }
        })
        .finally(() => {
            atualizacaoEmAndamento = false;
        });
}

function getBadgeClass(prioridade) {
    switch(prioridade.toLowerCase()) {
        case 'urgencia': return 'bg-danger';
        case 'normal': return 'bg-success';
        default: return 'bg-warning';
    }
}

function getPrioridadeTexto(prioridade) {
    const texto = prioridade.toUpperCase();
    return prioridade.toLowerCase() === 'urgencia' ? 'URGÊNCIA' : texto;
}

// Atualiza a cada 5 segundos
setInterval(atualizarFila, 5000);
</script>

<style>
    /* Estilos existentes... */

    /* Ajustes para Mobile */
    @media (max-width: 768px) {
        .card {
            margin-bottom: 15px;
        }
        
        .table-responsive {
            margin-bottom: 0;
            border: none;
        }
        
        .table {
            font-size: 14px; /* Texto um pouco menor para caber */
        }
        
        /* Transforma a tabela em cards para mobile */
        .table-mobile tr {
            display: block;
            margin-bottom: 15px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            padding: 12px;
        }
        
        .table-mobile td {
            display: block;
            border: none;
            padding: 8px 12px;
            text-align: left !important;
            position: relative;
        }
        
        /* Adiciona labels antes dos valores */
        .table-mobile td:before {
            content: attr(data-label);
            font-weight: bold;
            display: inline-block;
            width: 100%;
            margin-bottom: 4px;
        }
        
        /* Ajusta botões */
        .btn {
            width: 100%;
            margin: 5px 0;
            padding: 10px;
            font-size: 14px;
        }
        
        /* Ajusta o header da página */
        .page-header {
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .page-header h2 {
            font-size: 20px;
            margin: 0;
        }
        
        /* Melhora espaçamento */
        .container-fluid {
            padding: 15px;
        }
        
        /* Ajusta tamanho dos ícones */
        .fa, .fas {
            font-size: 16px;
        }
        
        /* Status com melhor visibilidade */
        .status-badge {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: bold;
            text-transform: uppercase;
            margin-top: 5px;
        }
        
        .status-aguardando {
            background-color: #ffd700;
            color: #000;
        }
        
        .status-chamado {
            background-color: #28a745;
            color: #fff;
        }
        
        .status-em_atendimento {
            background-color: #17a2b8;
            color: #fff;
        }
        
        .status-finalizado {
            background-color: #6c757d;
            color: #fff;
        }
        
        /* Ajusta campos de formulário */
        .form-control {
            font-size: 16px; /* Melhor para touch */
            height: 44px;
        }
        
        /* Melhora visualização de timestamps */
        .timestamp {
            font-size: 12px;
            color: #666;
            display: block;
            margin-top: 4px;
        }
    }

    /* Efeito do botão chamar */
    .btn-chamar {
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .btn-chamar-active {
        background-color: #28a745 !important;
        border-color: #28a745 !important;
        transform: scale(0.95);
        box-shadow: 0 0 15px rgba(40, 167, 69, 0.5);
    }
    
    .btn-chamar:active {
        transform: scale(0.95);
    }

    /* Ajustes específicos para mobile */
    @media (max-width: 768px) {
        .btn-chamar-active {
            transform: scale(0.98);
            box-shadow: 0 0 20px rgba(40, 167, 69, 0.7);
            animation: vibrate 0.3s ease-in-out;
        }
        
        /* Efeito de vibração sutil */
        @keyframes vibrate {
            0% { transform: translateX(0); }
            25% { transform: translateX(2px); }
            50% { transform: translateX(-2px); }
            75% { transform: translateX(2px); }
            100% { transform: translateX(0); }
        }
    }
</style>

<?php require_once '../../includes/footer.php'; ?> 