<?php
require_once '../../config/config.php';
require_once '../../includes/auth_admin.php';
require_once '../../includes/header.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

try {
    $database = new Database();
    $db = $database->getConnection();

    // Query mais simples para teste
    $query = "SELECT COUNT(*) as total FROM fila_atendimento";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "<!-- Total de registros: " . $result['total'] . " -->";

    // Agora a query completa
    $query = "SELECT 
                f.id, 
                f.tipo_atendimento, 
                f.prioridade, 
                f.status,
                f.horario_agendado,
                f.created_at,
                p.nome as paciente_nome, 
                p.cpf,
                u.nome as medico_nome,
                c.descricao as consultorio_nome
              FROM fila_atendimento f
              JOIN pacientes p ON f.paciente_id = p.id
              JOIN medicos m ON f.medico_id = m.id
              JOIN usuarios u ON m.usuario_id = u.id
              JOIN consultorios c ON f.consultorio_id = c.id
              WHERE f.status IN ('aguardando', 'chamado')
              ORDER BY 
                CASE 
                    WHEN f.tipo_atendimento = 'agendado' 
                         AND TIME(f.horario_agendado) <= CURRENT_TIME() 
                         AND TIME(f.horario_agendado) >= DATE_SUB(CURRENT_TIME(), INTERVAL 30 MINUTE) THEN 1
                    ELSE 2 
                END,
                CASE f.prioridade
                    WHEN 'urgencia' THEN 1
                    WHEN 'idoso' THEN 2
                    WHEN 'pcd' THEN 3
                    WHEN 'gestante' THEN 4
                    ELSE 5
                END,
                f.created_at ASC";

    echo "<!-- Query: " . $query . " -->";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    
    echo "<!-- Número de registros: " . $stmt->rowCount() . " -->";
    
    $fila = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<!-- Dados: " . print_r($fila, true) . " -->";

} catch (Exception $e) {
    echo "<!-- Erro: " . $e->getMessage() . " -->";
    echo "<!-- Arquivo: " . $e->getFile() . " -->";
    echo "<!-- Linha: " . $e->getLine() . " -->";
    die("Erro ao carregar a fila. Por favor, contate o administrador.");
}
?>

<div class="container mt-4">
    <!-- Filtro de Médicos -->
    <div class="row mb-3">
        <div class="col-md-4 ms-auto">
            <select class="form-select" id="filtroMedico">
                <option value="">Todos os Médicos</option>
                <?php
                $query_medicos = "SELECT DISTINCT m.id, u.nome 
                                FROM medicos m 
                                JOIN usuarios u ON m.usuario_id = u.id 
                                WHERE m.status = 'ativo'
                                ORDER BY u.nome";
                $stmt_medicos = $db->prepare($query_medicos);
                $stmt_medicos->execute();
                $medicos = $stmt_medicos->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($medicos as $medico) {
                    echo "<option value='{$medico['id']}'>{$medico['nome']}</option>";
                }
                ?>
            </select>
        </div>
    </div>

    <ul class="nav nav-tabs mb-4">
        <li class="nav-item">
            <a class="nav-link active" data-bs-toggle="tab" href="#aguardando">Aguardando</a>
        </li>
        <li class="nav-item">
            <a class="nav-link" data-bs-toggle="tab" href="#chamados">Chamados</a>
        </li>
    </ul>

    <div class="tab-content">
        <!-- Aba Aguardando -->
        <div class="tab-pane fade show active" id="aguardando">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Fila de Atendimento</h5>
                    <a href="recepcao.php" class="btn btn-primary">Adicionar Paciente</a>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width: 20%">Paciente</th>
                                    <th style="width: 12%">CPF</th>
                                    <th style="width: 15%">Médico</th>
                                    <th style="width: 10%">Consultório</th>
                                    <th style="width: 10%">Tipo</th>
                                    <th style="width: 10%">Horário</th>
                                    <th style="width: 8%">Prioridade</th>
                                    <th style="width: 8%">Status</th>
                                    <th style="width: 15%">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($fila)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">Nenhum paciente na fila</td>
                                    </tr>
                                <?php else: ?>
                                    <?php 
                                    $fila_aguardando = array_filter($fila, function($a) { 
                                        return $a['status'] === 'aguardando'; 
                                    });
                                    if (empty($fila_aguardando)): 
                                    ?>
                                        <tr>
                                            <td colspan="8" class="text-center">Nenhum paciente aguardando</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($fila_aguardando as $atendimento): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></td>
                                                <td><?php echo htmlspecialchars($atendimento['cpf']); ?></td>
                                                <td><?php echo htmlspecialchars($atendimento['medico_nome']); ?></td>
                                                <td><?php echo htmlspecialchars($atendimento['consultorio_nome']); ?></td>
                                                <td>
                                                    <?php 
                                                        echo $atendimento['tipo_atendimento'] === 'agendado' ? 
                                                            '<span class="btn btn-sm btn-primary disabled">Agendado</span>' : 
                                                            '<span class="btn btn-sm btn-secondary disabled">Chegada</span>';
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php 
                                                        if ($atendimento['tipo_atendimento'] === 'agendado') {
                                                            echo date('H:i', strtotime($atendimento['horario_agendado']));
                                                        } else {
                                                            echo date('H:i', strtotime($atendimento['created_at']));
                                                        }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php 
                                                        $btnClass = '';
                                                        $texto = strtoupper($atendimento['prioridade']);
                                                        
                                                        switch ($atendimento['prioridade']) {
                                                            case 'urgencia':
                                                                $btnClass = 'btn-danger';
                                                                $texto = 'URGÊNCIA';
                                                                break;
                                                            case 'normal':
                                                                $btnClass = 'btn-success';
                                                                $texto = 'NORMAL';
                                                                break;
                                                            default:
                                                                $btnClass = 'btn-warning';
                                                                $texto = strtoupper($atendimento['prioridade']);
                                                        }
                                                    ?>
                                                    <span class="btn btn-sm <?php echo $btnClass; ?> disabled">
                                                        <?php echo $texto; ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="btn btn-sm btn-info disabled">
                                                        <?php echo ucfirst($atendimento['status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="btn-group">
                                                        <button class="btn btn-warning btn-sm btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                                            <i class="fas fa-bullhorn"></i> Chamar
                                                        </button>
                                                        <button class="btn btn-info btn-sm" onclick="editarPacienteFila(<?php echo $atendimento['id']; ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn btn-danger btn-sm" onclick="removerDaFila(<?php echo $atendimento['id']; ?>)">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Aba Chamados -->
        <div class="tab-pane fade" id="chamados">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Pacientes Chamados</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width: 20%">Paciente</th>
                                    <th style="width: 15%">Médico</th>
                                    <th style="width: 10%">Consultório</th>
                                    <th style="width: 10%">Tipo</th>
                                    <th style="width: 10%">Horário</th>
                                    <th style="width: 8%">Prioridade</th>
                                    <th style="width: 8%">Status</th>
                                    <th style="width: 15%">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $fila_chamados = array_filter($fila, function($a) { 
                                    return $a['status'] === 'chamado'; 
                                });
                                if (empty($fila_chamados)): 
                                ?>
                                    <tr>
                                        <td colspan="8" class="text-center">Nenhum paciente chamado</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($fila_chamados as $atendimento): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($atendimento['paciente_nome']); ?></td>
                                            <td><?php echo htmlspecialchars($atendimento['medico_nome']); ?></td>
                                            <td><?php echo htmlspecialchars($atendimento['consultorio_nome']); ?></td>
                                            <td>
                                                <?php 
                                                    echo $atendimento['tipo_atendimento'] === 'agendado' ? 
                                                        '<span class="btn btn-sm btn-primary disabled">Agendado</span>' : 
                                                        '<span class="btn btn-sm btn-secondary disabled">Chegada</span>';
                                                ?>
                                            </td>
                                            <td>
                                                <?php 
                                                    if ($atendimento['tipo_atendimento'] === 'agendado') {
                                                        echo date('H:i', strtotime($atendimento['horario_agendado']));
                                                    } else {
                                                        echo date('H:i', strtotime($atendimento['created_at']));
                                                    }
                                                ?>
                                            </td>
                                            <td>
                                                <?php 
                                                    $btnClass = '';
                                                    $texto = strtoupper($atendimento['prioridade']);
                                                    
                                                    switch ($atendimento['prioridade']) {
                                                        case 'urgencia':
                                                            $btnClass = 'btn-danger';
                                                            $texto = 'URGÊNCIA';
                                                            break;
                                                        case 'normal':
                                                            $btnClass = 'btn-success';
                                                            $texto = 'NORMAL';
                                                            break;
                                                        default:
                                                            $btnClass = 'btn-warning';
                                                            $texto = strtoupper($atendimento['prioridade']);
                                                    }
                                                ?>
                                                <span class="btn btn-sm <?php echo $btnClass; ?> disabled">
                                                    <?php echo $texto; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="btn btn-sm btn-info disabled">
                                                    <?php echo ucfirst($atendimento['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button class="btn btn-warning btn-sm btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                                        <i class="fas fa-bullhorn"></i> Chamar
                                                    </button>
                                                    <button class="btn btn-info btn-sm" onclick="editarPacienteFila(<?php echo $atendimento['id']; ?>)">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button class="btn btn-danger btn-sm" onclick="removerDaFila(<?php echo $atendimento['id']; ?>)">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal de Edição -->
<div class="modal fade" id="editarFilaModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Editar Paciente na Fila</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="formEditarFila">
                    <input type="hidden" name="id" id="edit_id">
                    
                    <div class="mb-3">
                        <label class="form-label">Tipo de Atendimento</label>
                        <select class="form-control" name="tipo_atendimento" id="edit_tipo_atendimento" required>
                            <option value="chegada">Chegada</option>
                            <option value="agendado">Agendado</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Prioridade</label>
                        <select class="form-control" name="prioridade" id="edit_prioridade" required>
                            <option value="normal">Normal</option>
                            <option value="urgencia">Urgência</option>
                            <option value="idoso">Idoso</option>
                            <option value="pcd">PCD</option>
                            <option value="gestante">Gestante</option>
                        </select>
                    </div>

                    <div class="mb-3" id="div_horario_agendado" style="display:none;">
                        <label class="form-label">Horário Agendado</label>
                        <input type="time" class="form-control" name="horario_agendado" id="edit_horario_agendado">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Observações</label>
                        <textarea class="form-control" name="observacoes" id="edit_observacoes" rows="2"></textarea>
                    </div>

                    <button type="submit" class="btn btn-primary">Salvar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Remova os scripts do Select2 -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
const BASE_URL = '<?php echo BASE_URL; ?>';

// Função de atualização automática para a página da recepção
function atualizarFila() {
    if (atualizacaoEmAndamento) return;
    atualizacaoEmAndamento = true;
    
    const medicoSelecionado = $('#filtroMedico').find('option:selected').text().trim();
    
    fetch(`${BASE_URL}/src/admin/fila.php?action=buscar_fila&medico_nome=${encodeURIComponent(medicoSelecionado)}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const abaAtiva = document.querySelector('.tab-pane.active').id;
                const filaAguardando = data.pacientes.filter(p => p.status === 'aguardando');
                const filaChamados = data.pacientes.filter(p => p.status === 'chamado');
                
                // Atualiza a tabela de aguardando
                const tabelaAguardando = document.querySelector('#aguardando .table tbody');
                if (filaAguardando.length === 0) {
                    tabelaAguardando.innerHTML = '<tr><td colspan="8" class="text-center">Nenhum paciente na fila</td></tr>';
                } else {
                    tabelaAguardando.innerHTML = filaAguardando.map(atendimento => `
                        <tr>
                            <td>${atendimento.paciente_nome}</td>
                            <td>${atendimento.medico_nome}</td>
                            <td>${atendimento.consultorio_nome}</td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    '<span class="btn btn-sm btn-primary disabled">Agendado</span>' : 
                                    '<span class="btn btn-sm btn-secondary disabled">Chegada</span>'}
                            </td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    atendimento.horario_agendado : 
                                    new Date(atendimento.created_at).toLocaleTimeString().substr(0,5)}
                            </td>
                            <td>
                                <span class="btn btn-sm ${getPrioridadeClass(atendimento.prioridade)} disabled">
                                    ${atendimento.prioridade.toUpperCase()}
                                </span>
                            </td>
                            <td>
                                <span class="btn btn-sm btn-info disabled">Aguardando</span>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-warning btn-sm btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar
                                    </button>
                                    <button class="btn btn-info btn-sm" onclick="editarPacienteFila(${atendimento.id})">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="btn btn-danger btn-sm" onclick="removerDaFila(${atendimento.id})">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    `).join('');
                }

                // Atualiza a tabela de chamados
                const tabelaChamados = document.querySelector('#chamados .table tbody');
                if (filaChamados.length === 0) {
                    tabelaChamados.innerHTML = '<tr><td colspan="8" class="text-center">Nenhum paciente chamado</td></tr>';
                } else {
                    tabelaChamados.innerHTML = filaChamados.map(atendimento => `
                        <tr>
                            <td>${atendimento.paciente_nome}</td>
                            <td>${atendimento.medico_nome}</td>
                            <td>${atendimento.consultorio_nome}</td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    '<span class="btn btn-sm btn-primary disabled">Agendado</span>' : 
                                    '<span class="btn btn-sm btn-secondary disabled">Chegada</span>'}
                            </td>
                            <td>
                                ${atendimento.tipo_atendimento === 'agendado' ? 
                                    atendimento.horario_agendado : 
                                    new Date(atendimento.created_at).toLocaleTimeString().substr(0,5)}
                            </td>
                            <td>
                                <span class="btn btn-sm ${getPrioridadeClass(atendimento.prioridade)} disabled">
                                    ${atendimento.prioridade.toUpperCase()}
                                </span>
                            </td>
                            <td>
                                <span class="btn btn-sm btn-info disabled">Chamado</span>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-warning btn-sm btn-chamar" onclick="chamarPaciente(${atendimento.id}, this)">
                                        <i class="fas fa-bullhorn"></i> Chamar
                                    </button>
                                    <button class="btn btn-info btn-sm" onclick="editarPacienteFila(${atendimento.id})">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="btn btn-danger btn-sm" onclick="removerDaFila(${atendimento.id})">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    `).join('');
                }

                // Mantém a aba ativa
                const tab = new bootstrap.Tab(document.querySelector(`a[href="#${abaAtiva}"]`));
                tab.show();
            }
        })
        .finally(() => {
            atualizacaoEmAndamento = false;
        });
}

// Quando mudar o select, chama atualizarFila
$(document).ready(function() {
    $('#filtroMedico').on('change', function() {
        atualizarFila();
    });

    // Mostrar/esconder horário agendado baseado no tipo de atendimento
    $('#edit_tipo_atendimento').on('change', function() {
        if ($(this).val() === 'agendado') {
            $('#div_horario_agendado').show();
        } else {
            $('#div_horario_agendado').hide();
        }
    });

    // Form de edição
    $('#formEditarFila').on('submit', function(e) {
        e.preventDefault();
        const formData = new FormData(this);
        formData.append('action', 'editar');

        fetch(`${BASE_URL}/src/admin/fila.php`, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Erro ao atualizar paciente');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro ao processar requisição');
        });
    });
});

// Variável para controlar se já existe uma atualização em andamento
let atualizacaoEmAndamento = false;

function getPrioridadeClass(prioridade) {
    switch(prioridade.toLowerCase()) {
        case 'urgencia': return 'btn-danger';
        case 'normal': return 'btn-success';
        default: return 'btn-warning';
    }
}

// Atualiza a cada 5 segundos
setInterval(atualizarFila, 5000);

// Funções globais
function chamarPaciente(id, button) {
    // Adiciona classe de animação
    button.classList.add('btn-chamar-active');
    
    fetch(`${BASE_URL}/src/admin/fila.php`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=chamar&id=${id}`
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Atualiza a fila sem recarregar a página
            atualizarFila();
        } else {
            alert(data.message || 'Erro ao chamar paciente');
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        alert('Erro ao processar requisição');
    })
    .finally(() => {
        // Remove a classe após 1 segundo
        setTimeout(() => {
            button.classList.remove('btn-chamar-active');
        }, 1000);
    });
}

function editarPacienteFila(id) {
    fetch(`${BASE_URL}/src/admin/fila.php?action=read&id=${id}`)
    .then(response => response.json())
    .then(data => {
        $('#edit_id').val(data.id);
        $('#edit_tipo_atendimento').val(data.tipo_atendimento === 'normal' ? 'chegada' : data.tipo_atendimento);
        $('#edit_prioridade').val(data.prioridade);
        $('#edit_horario_agendado').val(data.horario_agendado ? data.horario_agendado.substr(11, 5) : '');
        $('#edit_observacoes').val(data.observacoes);
        
        if (data.tipo_atendimento === 'agendado') {
            $('#div_horario_agendado').show();
        } else {
            $('#div_horario_agendado').hide();
        }
        
        new bootstrap.Modal(document.getElementById('editarFilaModal')).show();
    });
}

function removerDaFila(id) {
    if (confirm('Deseja realmente remover este paciente da fila?')) {
        fetch(`${BASE_URL}/src/admin/fila.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=delete&id=${id}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Erro ao remover paciente');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro ao processar requisição');
        });
    }
}
</script>

<style>
    /* ... outros estilos ... */

    /* Efeito do botão chamar */
    .btn-chamar {
        transition: all 0.3s ease;
    }
    
    .btn-chamar-active {
        background-color: #28a745 !important;
        border-color: #28a745 !important;
        transform: scale(0.95);
        box-shadow: 0 0 15px rgba(40, 167, 69, 0.5);
    }
    
    .btn-chamar:active {
        transform: scale(0.95);
    }

    /* Efeito de pulsar */
    @keyframes pulse {
        0% {
            box-shadow: 0 0 0 0 rgba(255, 193, 7, 0.7);
        }
        70% {
            box-shadow: 0 0 0 10px rgba(255, 193, 7, 0);
        }
        100% {
            box-shadow: 0 0 0 0 rgba(255, 193, 7, 0);
        }
    }
</style>

<?php require_once '../../includes/footer.php'; ?> 