class FilaManager {
    constructor() {
        this.updateInterval = 5000; // 5 segundos
        this.initializeEventListeners();
        this.startAutoUpdate();
    }

    initializeEventListeners() {
        // Atualiza a fila quando a página carrega
        document.addEventListener('DOMContentLoaded', () => this.atualizarFila());

        // Delegação de eventos para botões da fila
        document.getElementById('filaPacientes').addEventListener('click', (e) => {
            if (e.target.classList.contains('btn-chamar')) {
                this.chamarPaciente(e.target.dataset.id);
            } else if (e.target.classList.contains('btn-iniciar')) {
                this.iniciarAtendimento(e.target.dataset.id);
            } else if (e.target.classList.contains('btn-finalizar')) {
                this.finalizarAtendimento(e.target.dataset.id);
            }
        });
    }

    startAutoUpdate() {
        setInterval(() => this.atualizarFila(), this.updateInterval);
    }

    async atualizarFila() {
        try {
            const response = await fetch('../../src/fila/get_fila.php');
            const data = await response.json();

            if (data.success) {
                this.renderizarFila(data.fila);
                this.atualizarEstatisticas(data.estatisticas);
            }
        } catch (error) {
            console.error('Erro ao atualizar fila:', error);
        }
    }

    renderizarFila(fila) {
        const tbody = document.getElementById('filaPacientes');
        tbody.innerHTML = '';

        fila.forEach(paciente => {
            const tr = document.createElement('tr');
            tr.innerHTML = `
                <td>${paciente.nome}</td>
                <td>${this.formatarHorario(paciente.created_at)}</td>
                <td>
                    ${paciente.prioridade ? 
                      '<span class="badge bg-warning">Prioritário</span>' : 
                      '<span class="badge bg-secondary">Normal</span>'}
                </td>
                <td>
                    ${this.renderizarBotoes(paciente)}
                </td>
            `;
            tbody.appendChild(tr);
        });
    }

    renderizarBotoes(paciente) {
        switch(paciente.status) {
            case 'aguardando':
                return `<button class="btn btn-warning btn-sm btn-chamar" data-id="${paciente.id}">
                            <i class="fas fa-bullhorn"></i> Chamar
                        </button>`;
            case 'chamado':
                return `<button class="btn btn-success btn-sm btn-iniciar" data-id="${paciente.id}">
                            <i class="fas fa-play"></i> Iniciar
                        </button>`;
            case 'em_atendimento':
                return `<button class="btn btn-danger btn-sm btn-finalizar" data-id="${paciente.id}">
                            <i class="fas fa-check"></i> Finalizar
                        </button>`;
            default:
                return '';
        }
    }

    async chamarPaciente(id) {
        try {
            const response = await fetch('../../src/fila/chamar_paciente.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ id })
            });

            const data = await response.json();
            if (data.success) {
                this.atualizarFila();
                this.reproduzirSom();
            }
        } catch (error) {
            console.error('Erro ao chamar paciente:', error);
        }
    }

    reproduzirSom() {
        const audio = new Audio('../../assets/sounds/chamada.mp3');
        audio.play();
    }

    formatarHorario(datetime) {
        return new Date(datetime).toLocaleTimeString('pt-BR', {
            hour: '2-digit',
            minute: '2-digit',
            timeZone: 'America/Sao_Paulo'
        });
    }

    atualizarEstatisticas(estatisticas) {
        document.getElementById('pacientesNaFila').textContent = estatisticas.na_fila;
        document.getElementById('atendimentosHoje').textContent = estatisticas.atendimentos_hoje;
        document.getElementById('tempoMedio').textContent = `${estatisticas.tempo_medio}min`;
    }
}

// Inicializa o gerenciador de fila
const filaManager = new FilaManager(); 