<?php
require_once '../../config/config.php';
require_once '../../config/database.php';

header('Content-Type: application/json');

// Verifica se é admin
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    die(json_encode(['success' => false, 'message' => 'Acesso negado']));
}

$database = new Database();
$db = $database->getConnection();

// Buscar paciente
if (isset($_GET['action']) && $_GET['action'] === 'buscar_paciente') {
    try {
        $busca = $_GET['busca'];
        
        // Remove caracteres especiais se for CPF
        $buscaSemMascara = preg_replace('/[^0-9]/', '', $busca);
        
        // Se for número com 11 dígitos, considera CPF
        if (strlen($buscaSemMascara) === 11) {
            // Formata o CPF
            $cpfFormatado = substr($buscaSemMascara, 0, 3) . '.' . 
                           substr($buscaSemMascara, 3, 3) . '.' . 
                           substr($buscaSemMascara, 6, 3) . '-' . 
                           substr($buscaSemMascara, 9, 2);
            
            $query = "SELECT id, nome, cpf FROM pacientes 
                     WHERE cpf = :cpf 
                     AND status = 'ativo'
                     LIMIT 1";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':cpf', $cpfFormatado);
        } else {
            // Busca por nome
            $query = "SELECT id, nome, cpf FROM pacientes 
                     WHERE nome LIKE :nome 
                     AND status = 'ativo'
                     ORDER BY nome
                     LIMIT 10";
            
            $stmt = $db->prepare($query);
            $stmt->bindValue(':nome', "%$busca%");
        }
        
        $stmt->execute();
        $pacientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (count($pacientes) > 0) {
            if (count($pacientes) === 1) {
                echo json_encode(['success' => true, 'paciente' => $pacientes[0]]);
            } else {
                echo json_encode(['success' => true, 'multiplos' => true, 'pacientes' => $pacientes]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Paciente não encontrado']);
        }
        
    } catch(PDOException $e) {
        error_log("Erro ao buscar paciente: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao buscar paciente']);
    }
    exit;
}

// Adicionar à fila
if (isset($_POST['action']) && $_POST['action'] === 'adicionar_fila') {
    try {
        // Verifica se paciente já está na fila deste médico
        $query = "SELECT id FROM fila_atendimento 
                 WHERE paciente_id = :paciente_id 
                 AND medico_id = :medico_id 
                 AND status IN ('aguardando', 'chamado')";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':paciente_id', $_POST['paciente_id']);
        $stmt->bindParam(':medico_id', $_POST['medico_id']);
        $stmt->execute();
        
        if ($stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'Paciente já está na fila deste médico']);
            exit;
        }

        $query = "INSERT INTO fila_atendimento (
            paciente_id, medico_id, consultorio_id, tipo_atendimento,
            horario_agendado, prioridade, observacoes, status
        ) VALUES (
            :paciente_id, :medico_id, :consultorio_id, :tipo_atendimento,
            :horario_agendado, :prioridade, :observacoes, 'aguardando'
        )";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':paciente_id', $_POST['paciente_id']);
        $stmt->bindParam(':medico_id', $_POST['medico_id']);
        $stmt->bindParam(':consultorio_id', $_POST['consultorio_id']);
        $stmt->bindParam(':tipo_atendimento', $_POST['tipo_atendimento']);
        $stmt->bindParam(':horario_agendado', $_POST['horario_agendado']);
        $stmt->bindParam(':prioridade', $_POST['prioridade']);
        $stmt->bindParam(':observacoes', $_POST['observacoes']);
        
        $stmt->execute();
        echo json_encode(['success' => true, 'message' => 'Paciente adicionado à fila com sucesso']);

    } catch(PDOException $e) {
        error_log("Erro ao adicionar à fila: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao adicionar paciente à fila']);
    }
    exit;
}

// Chamar paciente
if (isset($_POST['action']) && $_POST['action'] === 'chamar') {
    try {
        // Atualiza o status e horário da chamada
        $query = "UPDATE fila_atendimento 
                 SET status = 'chamado', 
                     called_at = NOW() 
                 WHERE id = :id";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_POST['id']);
        $result = $stmt->execute();
        
        if ($result) {
            // Busca os dados da chamada
            $query = "SELECT 
                        f.id,
                        p.nome as paciente_nome,
                        c.descricao as consultorio_descricao,
                        u.nome as medico_nome,
                        m.foto as medico_foto,
                        f.called_at
                     FROM fila_atendimento f
                     JOIN pacientes p ON f.paciente_id = p.id
                     JOIN medicos m ON f.medico_id = m.id
                     JOIN usuarios u ON m.usuario_id = u.id
                     JOIN consultorios c ON f.consultorio_id = c.id
                     WHERE f.id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $_POST['id']);
            $stmt->execute();
            $chamada = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Marca que houve uma nova chamada
            $_SESSION['nova_chamada'] = true;
            
            echo json_encode([
                'success' => true,
                'message' => 'Paciente chamado com sucesso',
                'chamada' => $chamada
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao chamar paciente']);
        }
        
    } catch(PDOException $e) {
        error_log("Erro ao chamar paciente: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao chamar paciente']);
    }
    exit;
}

// Remover da fila
if (isset($_POST['action']) && $_POST['action'] === 'remover') {
    try {
        $query = "UPDATE fila_atendimento 
                 SET status = 'cancelado' 
                 WHERE id = :id";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_POST['id']);
        $stmt->execute();
        
        echo json_encode(['success' => true, 'message' => 'Paciente removido da fila com sucesso']);
        
    } catch(PDOException $e) {
        error_log("Erro ao remover paciente: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao remover paciente da fila']);
    }
    exit;
}

// Buscar paciente da fila
if (isset($_GET['action']) && $_GET['action'] === 'buscar' && isset($_GET['id'])) {
    try {
        // Alterando a query para buscar todos os campos necessários
        $query = "SELECT 
                    f.id,
                    f.tipo_atendimento,
                    f.horario_agendado,
                    f.prioridade,
                    f.observacoes,
                    f.status,
                    p.nome as paciente_nome,
                    p.cpf,
                    u.nome as medico_nome,
                    c.descricao as consultorio_nome
                FROM fila_atendimento f
                JOIN pacientes p ON f.paciente_id = p.id
                JOIN medicos m ON f.medico_id = m.id
                JOIN usuarios u ON m.usuario_id = u.id
                JOIN consultorios c ON f.consultorio_id = c.id
                WHERE f.id = :id";

        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_GET['id']);
        $stmt->execute();
        
        $paciente = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($paciente) {
            // Debug
            error_log("Dados do paciente encontrados: " . print_r($paciente, true));
            echo json_encode(['success' => true, 'paciente' => $paciente]);
        } else {
            error_log("Paciente não encontrado na fila. ID: " . $_GET['id']);
            echo json_encode(['success' => false, 'message' => 'Paciente não encontrado na fila']);
        }
        
    } catch(PDOException $e) {
        error_log("Erro ao buscar paciente da fila: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao buscar paciente']);
    }
    exit;
}

// Editar paciente na fila
if (isset($_POST['action']) && $_POST['action'] === 'editar') {
    try {
        // Debug para ver os dados recebidos
        error_log("Dados recebidos para edição: " . print_r($_POST, true));

        // Validar dados recebidos
        if (!isset($_POST['id']) || !isset($_POST['prioridade'])) {
            throw new Exception("Dados incompletos");
        }

        // Query mais simples, apenas atualizando a prioridade
        $query = "UPDATE fila_atendimento 
                 SET prioridade = :prioridade 
                 WHERE id = :id";
            
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $_POST['id']);
        $stmt->bindParam(':prioridade', $_POST['prioridade']);
        
        $result = $stmt->execute();
        
        if ($result) {
            error_log("Atualização bem sucedida para ID: " . $_POST['id'] . " - Nova prioridade: " . $_POST['prioridade']);
            echo json_encode(['success' => true, 'message' => 'Paciente atualizado com sucesso']);
        } else {
            error_log("Erro ao executar update: " . print_r($stmt->errorInfo(), true));
            echo json_encode(['success' => false, 'message' => 'Erro ao atualizar paciente']);
        }
        
    } catch(Exception $e) {
        error_log("Erro ao editar paciente na fila: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erro ao atualizar paciente: ' . $e->getMessage()]);
    }
    exit;
} 