<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Verificar login
if (!isLoggedIn() || $_SESSION['user']['type'] !== 'admin') {
    header("Location: ../index.php");
    exit();
}

$message = '';
$doctor = null;
$schedule = [];

if (isset($_GET['id'])) {
    $doctor = getDoctorById($_GET['id']);
    $schedule = getDoctorSchedule($_GET['id']);
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $doctorId = $_POST['doctor_id'];
        $weekdays = $_POST['weekdays'] ?? [];
        
        if (updateDoctorSchedule($doctorId, $weekdays)) {
            $message = "Horários atualizados com sucesso!";
            $schedule = getDoctorSchedule($doctorId);
        }
    } catch (Exception $e) {
        $message = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dias de Atendimento</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        .weekday-btn {
            min-width: 100px;
            padding: 12px 24px;
            border-radius: 0.5rem;
            font-size: 1.125rem;
            font-weight: 600;
            color: white;
            cursor: pointer;
            margin: 0 6px;
            transition: all 0.2s;
        }
        .bg-yellow-custom {
            background-color: #FDB813;
        }
        .bg-yellow-custom:hover {
            background-color: #e5a712;
        }
        .bg-red-custom {
            background-color: #FF6B6B;
        }
        .bg-red-custom:hover {
            background-color: #ff5252;
        }
        .bg-cyan-custom {
            background-color: #06B6D4;
        }
        .bg-cyan-custom:hover {
            background-color: #0891b2;
        }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8 max-w-5xl">
        <?php if ($doctor): ?>
            <div class="bg-white rounded-lg shadow-md p-8">
                <h1 class="text-2xl text-purple-600 mb-6">
                    DIA DE ATENDIMENTO PARA MÉDICO(A) <?= strtoupper(htmlspecialchars($doctor['name'])) ?>
                </h1>
                
                <div class="text-gray-700 mb-8 text-lg">
                    <p>
                        Especialidade: <strong class="text-gray-800"><?= htmlspecialchars($doctor['specialty_name']) ?></strong>
                        Local: <strong class="text-gray-800"><?= htmlspecialchars($doctor['office_name']) ?></strong>
                    </p>
                </div>

                <div class="flex justify-center gap-4 mb-8">
                    <?php
                    $weekdays = ['Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sab', 'Dom'];
                    
                    foreach ($weekdays as $day):
                        $isActive = isset($schedule[$day]);
                        
                        switch ($day) {
                            case 'Dom':
                            case 'Sab':
                                $baseColor = 'bg-yellow-custom';
                                break;
                            case 'Qua':
                                $baseColor = 'bg-red-custom';
                                break;
                            default:
                                $baseColor = 'bg-cyan-custom';
                                break;
                        }
                    ?>
                        <button type="button" 
                                class="weekday-btn <?= $baseColor ?> shadow-sm"
                                data-day="<?= $day ?>">
                            <?= $day ?>
                        </button>
                    <?php endforeach; ?>
                </div>

                <div class="flex items-center justify-center gap-8 text-base">
                    <div class="flex items-center">
                        <div class="w-4 h-4 bg-red-custom rounded mr-3"></div>
                        <span>Dias com horários já Cadastrado</span>
                    </div>
                    <div class="flex items-center">
                        <div class="w-4 h-4 bg-cyan-custom rounded mr-3"></div>
                        <span>Dias sem horários Cadastrado</span>
                    </div>
                </div>
            </div>

            <form method="post" id="scheduleForm" class="hidden">
                <input type="hidden" name="doctor_id" value="<?= $doctor['id'] ?>">
                <?php foreach ($weekdays as $day): ?>
                    <input type="hidden" name="weekdays[<?= $day ?>]" value="<?= isset($schedule[$day]) ? '1' : '0' ?>" 
                           class="weekday-input" data-day="<?= $day ?>">
                <?php endforeach; ?>
            </form>
        <?php else: ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative">
                Médico não encontrado.
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
    <script>
        $(document).ready(function() {
            $('.weekday-btn').click(function() {
                const btn = $(this);
                const day = btn.data('day');
                const input = $(`input[data-day="${day}"]`);
                const isActive = input.val() === '1';
                
                if (day !== 'Sab' && day !== 'Dom') {
                    if (isActive) {
                        btn.removeClass('bg-red-custom')
                           .addClass('bg-cyan-custom');
                        input.val('0');
                    } else {
                        btn.removeClass('bg-cyan-custom')
                           .addClass('bg-red-custom');
                        input.val('1');
                    }
                }
                
                // Enviar formulário automaticamente
                $('#scheduleForm').submit();
            });
        });
    </script>
</body>
</html> 