/**
 * CliniSys - Módulo da TV Sala de Espera
 * Gerencia chamadas em tempo real com voz e vídeo
 */

class TVManager {
    constructor() {
        this.pollingInterval = 3000; // 3 segundos para TV
        this.pollingTimer = null;
        this.currentCalls = [];
        this.displayedCallIds = [];
        this.speechEnabled = true;
        this.maxCallsDisplay = 5;
        this.callDisplayTime = 30000; // 30 segundos
        this.apiBase = 'api/';
        
        this.init();
    }
    
    init() {
        this.setupSpeechSynthesis();
        this.startClock();
        this.loadInitialData();
        this.startPolling();
        this.setupEventHandlers();
        
        console.log('TV Manager inicializado');
    }
    
    setupSpeechSynthesis() {
        // Verificar se o navegador suporta síntese de voz
        if ('speechSynthesis' in window) {
            this.speechSynthesis = window.speechSynthesis;
            this.speechEnabled = true;
            
            // Configurar voz em português
            this.setupPortugueseVoice();
        } else {
            console.warn('Síntese de voz não suportada neste navegador');
            this.speechEnabled = false;
        }
    }
    
    setupPortugueseVoice() {
        // Aguardar carregamento das vozes
        const setVoice = () => {
            const voices = this.speechSynthesis.getVoices();
            
            // Procurar voz em português
            this.voice = voices.find(voice => 
                voice.lang.startsWith('pt') || 
                voice.name.toLowerCase().includes('portuguese') ||
                voice.name.toLowerCase().includes('brasil')
            ) || voices[0]; // Fallback para primeira voz disponível
            
            if (this.voice) {
                console.log('Voz selecionada:', this.voice.name, this.voice.lang);
            }
        };
        
        if (this.speechSynthesis.getVoices().length > 0) {
            setVoice();
        } else {
            this.speechSynthesis.addEventListener('voiceschanged', setVoice);
        }
    }
    
    setupEventHandlers() {
        // Teclas de controle
        document.addEventListener('keydown', (e) => {
            switch (e.key) {
                case 'F5':
                    e.preventDefault();
                    this.refreshData();
                    break;
                case 'F11':
                    this.toggleFullscreen();
                    break;
                case 'Escape':
                    if (document.fullscreenElement) {
                        document.exitFullscreen();
                    }
                    break;
                case 'm':
                case 'M':
                    this.toggleSpeech();
                    break;
            }
        });
        
        // Detectar quando a página fica visível/invisível
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.stopPolling();
            } else {
                this.startPolling();
            }
        });
        
        // Recarregar quando a conexão for restaurada
        window.addEventListener('online', () => {
            this.refreshData();
        });
    }
    
    startClock() {
        this.updateClock();
        setInterval(() => this.updateClock(), 1000);
    }
    
    updateClock() {
        const now = new Date();
        const timeString = now.toLocaleString('pt-BR', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit'
        });
        
        document.getElementById('tv-clock').textContent = timeString;
    }
    
    async loadInitialData() {
        try {
            await this.loadCalls();
        } catch (error) {
            console.error('Erro ao carregar dados iniciais:', error);
            this.showError('Erro de conexão. Tentando reconectar...');
        }
    }
    
    async loadCalls() {
        try {
            const response = await this.apiCall('fila.php?action=tv-calls');
            
            if (response.success) {
                const newCalls = response.data;
                this.processCalls(newCalls);
            } else {
                console.error('Erro na resposta da API:', response.message);
            }
        } catch (error) {
            console.error('Erro ao carregar chamadas:', error);
            throw error;
        }
    }
    
    processCalls(newCalls) {
        // Identificar novas chamadas
        const newCallIds = newCalls.map(call => call.id);
        const currentCallIds = this.currentCalls.map(call => call.id);
        
        const addedCalls = newCalls.filter(call => !currentCallIds.includes(call.id));
        const removedCallIds = currentCallIds.filter(id => !newCallIds.includes(id));
        
        // Processar novas chamadas
        addedCalls.forEach(call => {
            this.announceCall(call);
        });
        
        // Atualizar lista atual
        this.currentCalls = newCalls;
        
        // Atualizar display
        this.updateCallsDisplay();
        
        // Marcar chamadas como exibidas
        if (newCallIds.length > 0) {
            this.markCallsAsShown(newCallIds);
        }
    }
    
    updateCallsDisplay() {
        const container = document.getElementById('tv-chamadas');
        
        if (this.currentCalls.length === 0) {
            container.innerHTML = this.getEmptyState();
            return;
        }
        
        // Limitar número de chamadas exibidas
        const callsToShow = this.currentCalls.slice(0, this.maxCallsDisplay);
        
        container.innerHTML = '';
        
        callsToShow.forEach((call, index) => {
            const callElement = this.createCallElement(call, index);
            container.appendChild(callElement);
        });
    }
    
    createCallElement(call, index) {
        const div = document.createElement('div');
        div.className = `call-item ${this.getPriorityClass(call)} new-call`;
        div.style.animationDelay = `${index * 0.2}s`;
        
        div.innerHTML = `
            <div class="call-type">${this.getCallTypeText(call.tipo_chamada)}</div>
            <div class="call-number">SENHA ${call.numero_senha}</div>
            <div class="call-patient">${call.nome_paciente}</div>
            <div class="call-doctor">
                <i class="fas fa-user-md me-2"></i>
                ${call.medico}
            </div>
            <div class="call-location">
                <i class="fas fa-map-marker-alt me-2"></i>
                ${call.consultorio}
            </div>
        `;
        
        // Remover classe de animação após a animação
        setTimeout(() => {
            div.classList.remove('new-call');
        }, 1000);
        
        return div;
    }
    
    getPriorityClass(call) {
        // Assumindo que a prioridade pode ser inferida do tipo de chamada
        if (call.tipo_chamada === 'terceira') {
            return 'urgent';
        }
        return '';
    }
    
    getCallTypeText(tipo) {
        const tipos = {
            'primeira': '1ª Chamada',
            'segunda': '2ª Chamada',
            'terceira': '3ª Chamada'
        };
        return tipos[tipo] || 'Chamada';
    }
    
    getEmptyState() {
        return `
            <div class="empty-calls">
                <i class="fas fa-clock"></i>
                <h3>Aguardando Chamadas</h3>
                <p>Nenhuma chamada no momento</p>
            </div>
        `;
    }
    
    announceCall(call) {
        if (!this.speechEnabled || !this.speechSynthesis) {
            return;
        }
        
        // Parar qualquer anúncio anterior
        this.speechSynthesis.cancel();
        
        // Criar texto do anúncio
        const announcement = this.createAnnouncement(call);
        
        // Criar utterance
        const utterance = new SpeechSynthesisUtterance(announcement);
        
        // Configurar voz
        if (this.voice) {
            utterance.voice = this.voice;
        }
        
        // Configurações de voz
        utterance.rate = 0.8; // Velocidade mais lenta para clareza
        utterance.pitch = 1.0;
        utterance.volume = 0.8;
        
        // Eventos
        utterance.onstart = () => {
            console.log('Iniciando anúncio:', announcement);
        };
        
        utterance.onend = () => {
            console.log('Anúncio finalizado');
        };
        
        utterance.onerror = (event) => {
            console.error('Erro no anúncio:', event.error);
        };
        
        // Falar
        this.speechSynthesis.speak(utterance);
        
        // Repetir anúncio após 5 segundos se for segunda ou terceira chamada
        if (call.tipo_chamada === 'segunda' || call.tipo_chamada === 'terceira') {
            setTimeout(() => {
                if (this.speechEnabled) {
                    const repeatUtterance = new SpeechSynthesisUtterance(announcement);
                    if (this.voice) repeatUtterance.voice = this.voice;
                    repeatUtterance.rate = 0.8;
                    repeatUtterance.pitch = 1.0;
                    repeatUtterance.volume = 0.8;
                    this.speechSynthesis.speak(repeatUtterance);
                }
            }, 5000);
        }
    }
    
    createAnnouncement(call) {
        const tipoTexto = {
            'primeira': '',
            'segunda': 'Segunda chamada. ',
            'terceira': 'Terceira e última chamada. '
        };
        
        const prefixo = tipoTexto[call.tipo_chamada] || '';
        
        return `${prefixo}Senha ${call.numero_senha}, ${call.nome_paciente}, dirija-se ao ${call.consultorio} para consulta com doutor ${call.medico}.`;
    }
    
    async markCallsAsShown(callIds) {
        try {
            await this.apiCall('fila.php?action=mark-calls-shown', 'POST', {
                ids: callIds
            });
        } catch (error) {
            console.error('Erro ao marcar chamadas como exibidas:', error);
        }
    }
    
    startPolling() {
        if (this.pollingTimer) {
            clearInterval(this.pollingTimer);
        }
        
        this.pollingTimer = setInterval(() => {
            this.loadCalls().catch(error => {
                console.error('Erro no polling:', error);
            });
        }, this.pollingInterval);
    }
    
    stopPolling() {
        if (this.pollingTimer) {
            clearInterval(this.pollingTimer);
            this.pollingTimer = null;
        }
    }
    
    async refreshData() {
        try {
            await this.loadCalls();
            this.showMessage('Dados atualizados', 'success');
        } catch (error) {
            this.showError('Erro ao atualizar dados');
        }
    }
    
    toggleFullscreen() {
        if (!document.fullscreenElement) {
            document.documentElement.requestFullscreen().catch(err => {
                console.error('Erro ao entrar em tela cheia:', err);
            });
        } else {
            document.exitFullscreen();
        }
    }
    
    toggleSpeech() {
        this.speechEnabled = !this.speechEnabled;
        
        if (!this.speechEnabled) {
            this.speechSynthesis.cancel();
        }
        
        this.showMessage(
            this.speechEnabled ? 'Áudio habilitado' : 'Áudio desabilitado',
            this.speechEnabled ? 'success' : 'warning'
        );
    }
    
    showMessage(message, type = 'info') {
        // Criar toast temporário
        const toast = document.createElement('div');
        toast.className = `alert alert-${type} position-fixed`;
        toast.style.cssText = `
            top: 20px;
            right: 20px;
            z-index: 9999;
            min-width: 300px;
            animation: fadeIn 0.3s ease-in-out;
        `;
        toast.textContent = message;
        
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.style.animation = 'fadeOut 0.3s ease-in-out';
            setTimeout(() => toast.remove(), 300);
        }, 3000);
    }
    
    showError(message) {
        this.showMessage(message, 'danger');
    }
    
    async apiCall(endpoint, method = 'GET', data = null) {
        const url = this.apiBase + endpoint;
        const options = {
            method,
            headers: {
                'Content-Type': 'application/json'
            }
        };
        
        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }
        
        const response = await fetch(url, options);
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        return await response.json();
    }
}

// Extensão para preview na interface principal
class TVPreviewManager {
    constructor() {
        this.pollingInterval = 5000;
        this.pollingTimer = null;
        this.init();
    }
    
    init() {
        this.startClock();
        this.loadPreviewData();
        this.startPolling();
    }
    
    startClock() {
        this.updateClock();
        setInterval(() => this.updateClock(), 1000);
    }
    
    updateClock() {
        const clockElement = document.getElementById('tv-clock');
        if (clockElement) {
            const now = new Date();
            clockElement.textContent = now.toLocaleString('pt-BR');
        }
    }
    
    async loadPreviewData() {
        try {
            const response = await fetch('api/fila.php?action=tv-calls');
            const result = await response.json();
            
            if (result.success) {
                this.updatePreview(result.data);
            }
        } catch (error) {
            console.error('Erro ao carregar preview:', error);
        }
    }
    
    updatePreview(calls) {
        const container = document.getElementById('tv-chamadas');
        if (!container) return;
        
        if (calls.length === 0) {
            container.innerHTML = `
                <div class="text-center text-muted">
                    <i class="fas fa-clock fa-3x mb-3"></i>
                    <h4>Aguardando Chamadas</h4>
                    <p>Nenhuma chamada no momento</p>
                </div>
            `;
            return;
        }
        
        container.innerHTML = '';
        
        calls.slice(0, 3).forEach(call => {
            const div = document.createElement('div');
            div.className = 'chamada-item mb-3 p-3 bg-primary text-white rounded';
            div.innerHTML = `
                <div class="row align-items-center">
                    <div class="col-md-3">
                        <h3 class="mb-0">SENHA ${call.numero_senha}</h3>
                    </div>
                    <div class="col-md-6">
                        <strong>${call.nome_paciente}</strong><br>
                        <small>${call.medico}</small>
                    </div>
                    <div class="col-md-3">
                        <i class="fas fa-map-marker-alt me-1"></i>
                        ${call.consultorio}
                    </div>
                </div>
            `;
            container.appendChild(div);
        });
    }
    
    startPolling() {
        this.pollingTimer = setInterval(() => {
            this.loadPreviewData();
        }, this.pollingInterval);
    }
    
    stopPolling() {
        if (this.pollingTimer) {
            clearInterval(this.pollingTimer);
            this.pollingTimer = null;
        }
    }
}

// Inicialização
document.addEventListener('DOMContentLoaded', () => {
    // Verificar se estamos na página da TV ou no preview
    if (window.location.pathname.includes('tv.html') || document.title.includes('TV Sala de Espera')) {
        window.tvManager = new TVManager();
        
        // Entrar em tela cheia automaticamente após 3 segundos
        setTimeout(() => {
            if (!document.fullscreenElement) {
                document.documentElement.requestFullscreen().catch(err => {
                    console.log('Tela cheia não disponível:', err);
                });
            }
        }, 3000);
    } else {
        // Preview na interface principal
        window.tvPreviewManager = new TVPreviewManager();
    }
});

// Extensões para o app principal
if (window.app) {
    window.app.loadTVData = async function() {
        if (window.tvPreviewManager) {
            await window.tvPreviewManager.loadPreviewData();
        }
    };
}

