/**
 * CliniSys - Módulo de Fila de Espera
 * Gerencia a fila de espera com atualizações em tempo real
 */

class QueueManager {
    constructor() {
        this.pollingInterval = 5000; // 5 segundos
        this.pollingTimer = null;
        this.isPolling = false;
        this.currentQueue = [];
        this.audioEnabled = true;
        this.init();
    }
    
    init() {
        this.setupEventHandlers();
        this.loadInitialData();
    }
    
    setupEventHandlers() {
        // Botão chamar próximo
        document.querySelector('[data-action="chamar-proximo"]')?.addEventListener('click', () => {
            this.callNext();
        });
        
        // Botão adicionar à fila
        document.querySelector('[data-action="adicionar-fila"]')?.addEventListener('click', () => {
            this.showAddToQueueModal();
        });
        
        // Eventos de ação na tabela da fila
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-queue-action]')) {
                const action = e.target.dataset.queueAction;
                const filaId = e.target.dataset.filaId;
                this.handleQueueAction(action, filaId);
            }
        });
        
        // Configurações de áudio
        const audioToggle = document.getElementById('audio-toggle');
        if (audioToggle) {
            audioToggle.addEventListener('change', (e) => {
                this.audioEnabled = e.target.checked;
                localStorage.setItem('clinisys_audio_enabled', this.audioEnabled);
            });
            
            // Carregar configuração salva
            const savedAudioSetting = localStorage.getItem('clinisys_audio_enabled');
            if (savedAudioSetting !== null) {
                this.audioEnabled = savedAudioSetting === 'true';
                audioToggle.checked = this.audioEnabled;
            }
        }
    }
    
    async loadInitialData() {
        try {
            await this.loadQueueData();
            await this.loadQueueStats();
        } catch (error) {
            console.error('Erro ao carregar dados iniciais da fila:', error);
        }
    }
    
    async loadQueueData() {
        try {
            const response = await window.app?.apiCall('fila.php?action=current');
            if (response?.success) {
                this.currentQueue = response.data;
                this.updateQueueTable();
                this.updateQueueCounts();
            }
        } catch (error) {
            console.error('Erro ao carregar dados da fila:', error);
            window.app?.showError('Erro ao carregar fila de espera');
        }
    }
    
    async loadQueueStats() {
        try {
            const response = await window.app?.apiCall('fila.php?action=stats');
            if (response?.success) {
                this.updateQueueStats(response.data);
            }
        } catch (error) {
            console.error('Erro ao carregar estatísticas da fila:', error);
        }
    }
    
    updateQueueTable() {
        const tbody = document.querySelector('#fila-table tbody');
        if (!tbody) return;
        
        tbody.innerHTML = '';
        
        if (this.currentQueue.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="8" class="text-center text-muted">
                        <i class="fas fa-inbox fa-2x mb-2"></i><br>
                        Nenhum paciente na fila
                    </td>
                </tr>
            `;
            return;
        }
        
        this.currentQueue.forEach(item => {
            const row = document.createElement('tr');
            row.className = `queue-item priority-${item.prioridade}`;
            row.innerHTML = `
                <td>
                    <span class="badge bg-primary fs-6">${item.numero_senha}</span>
                </td>
                <td>
                    <strong>${item.paciente_nome}</strong>
                    ${item.paciente_convenio ? `<br><small class="text-muted">${item.paciente_convenio}</small>` : ''}
                </td>
                <td>
                    ${item.medico_nome}
                    ${item.especialidade_nome ? `<br><small class="text-muted">${item.especialidade_nome}</small>` : ''}
                </td>
                <td>${this.formatTime(item.data_chegada)}</td>
                <td>
                    <span class="badge ${this.getWaitTimeBadgeClass(item.tempo_espera)}">
                        ${item.tempo_espera} min
                    </span>
                </td>
                <td>
                    <span class="badge prioridade-${item.prioridade}">
                        ${this.formatPriority(item.prioridade)}
                    </span>
                </td>
                <td>
                    <span class="badge status-${item.status}">
                        ${this.formatStatus(item.status)}
                    </span>
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        ${this.getActionButtons(item)}
                    </div>
                </td>
            `;
            tbody.appendChild(row);
        });
    }
    
    updateQueueCounts() {
        const aguardando = this.currentQueue.filter(item => item.status === 'aguardando').length;
        const chamados = this.currentQueue.filter(item => item.status === 'chamado').length;
        const atendendo = this.currentQueue.filter(item => item.status === 'atendendo').length;
        
        document.getElementById('fila-aguardando').textContent = aguardando;
        document.getElementById('fila-chamados').textContent = chamados;
        document.getElementById('fila-atendendo').textContent = atendendo;
        
        // Atualizar também no dashboard se estivermos lá
        if (window.app?.currentSection === 'dashboard') {
            document.getElementById('stats-fila-espera').textContent = aguardando;
        }
    }
    
    updateQueueStats(stats) {
        document.getElementById('fila-tempo-medio').textContent = 
            stats.tempo_medio_espera ? Math.round(stats.tempo_medio_espera) + ' min' : '0 min';
    }
    
    getActionButtons(item) {
        const buttons = [];
        
        switch (item.status) {
            case 'aguardando':
                buttons.push(`
                    <button class="btn btn-success" data-queue-action="call" data-fila-id="${item.id}" title="Chamar">
                        <i class="fas fa-bullhorn"></i>
                    </button>
                `);
                buttons.push(`
                    <button class="btn btn-warning" data-queue-action="priority" data-fila-id="${item.id}" title="Alterar Prioridade">
                        <i class="fas fa-exclamation"></i>
                    </button>
                `);
                break;
                
            case 'chamado':
                buttons.push(`
                    <button class="btn btn-primary" data-queue-action="confirm" data-fila-id="${item.id}" title="Confirmar Atendimento">
                        <i class="fas fa-check"></i>
                    </button>
                `);
                buttons.push(`
                    <button class="btn btn-warning" data-queue-action="recall" data-fila-id="${item.id}" title="Chamar Novamente">
                        <i class="fas fa-redo"></i>
                    </button>
                `);
                break;
                
            case 'atendendo':
                buttons.push(`
                    <button class="btn btn-success" data-queue-action="finish" data-fila-id="${item.id}" title="Finalizar Atendimento">
                        <i class="fas fa-check-circle"></i>
                    </button>
                `);
                break;
        }
        
        // Botão remover sempre disponível (exceto para finalizados)
        if (item.status !== 'finalizado') {
            buttons.push(`
                <button class="btn btn-danger" data-queue-action="remove" data-fila-id="${item.id}" title="Remover da Fila">
                    <i class="fas fa-times"></i>
                </button>
            `);
        }
        
        return buttons.join('');
    }
    
    async handleQueueAction(action, filaId) {
        try {
            let result;
            
            switch (action) {
                case 'call':
                    result = await this.callSpecific(filaId);
                    break;
                case 'confirm':
                    result = await this.confirmAttendance(filaId);
                    break;
                case 'finish':
                    result = await this.finishAttendance(filaId);
                    break;
                case 'recall':
                    result = await this.recallPatient(filaId);
                    break;
                case 'priority':
                    await this.showPriorityModal(filaId);
                    return;
                case 'remove':
                    if (confirm('Tem certeza que deseja remover este paciente da fila?')) {
                        result = await this.removeFromQueue(filaId);
                    }
                    return;
            }
            
            if (result?.success) {
                window.app?.showSuccess(result.message);
                await this.loadQueueData();
            } else {
                window.app?.showError(result?.message || 'Erro na operação');
            }
        } catch (error) {
            console.error('Erro na ação da fila:', error);
            window.app?.showError('Erro ao executar ação');
        }
    }
    
    async callNext() {
        try {
            window.app?.showLoading();
            
            const consultorio = await this.getConsultorio();
            
            const response = await window.app?.apiCall('fila.php?action=call-next', 'POST', {
                consultorio: consultorio
            });
            
            if (response?.success) {
                window.app?.showSuccess(`Paciente ${response.paciente.paciente_nome} foi chamado!`);
                
                // Tocar som de chamada se habilitado
                if (this.audioEnabled) {
                    this.playCallSound();
                }
                
                await this.loadQueueData();
            } else {
                window.app?.showError(response?.message || 'Nenhum paciente na fila');
            }
        } catch (error) {
            console.error('Erro ao chamar próximo paciente:', error);
            window.app?.showError('Erro ao chamar próximo paciente');
        } finally {
            window.app?.hideLoading();
        }
    }
    
    async callSpecific(filaId) {
        const consultorio = await this.getConsultorio();
        
        // Primeiro atualizar status para chamado
        await window.app?.apiCall('fila.php?action=update-status', 'PUT', {
            fila_id: filaId,
            status: 'chamado'
        });
        
        // Depois criar chamada para TV
        return await window.app?.apiCall('fila.php?action=recall', 'POST', {
            fila_id: filaId,
            consultorio: consultorio
        });
    }
    
    async confirmAttendance(filaId) {
        return await window.app?.apiCall('fila.php?action=confirm-attendance', 'POST', {
            fila_id: filaId
        });
    }
    
    async finishAttendance(filaId) {
        return await window.app?.apiCall('fila.php?action=finish-attendance', 'POST', {
            fila_id: filaId
        });
    }
    
    async recallPatient(filaId) {
        const consultorio = await this.getConsultorio();
        
        return await window.app?.apiCall('fila.php?action=recall', 'POST', {
            fila_id: filaId,
            consultorio: consultorio
        });
    }
    
    async removeFromQueue(filaId) {
        const motivo = prompt('Motivo da remoção (opcional):') || 'Removido da fila';
        
        return await window.app?.apiCall(`fila.php?action=remove&fila_id=${filaId}&motivo=${encodeURIComponent(motivo)}`, 'DELETE');
    }
    
    async getConsultorio() {
        // Por enquanto retorna um valor padrão, mas pode ser expandido para seleção
        return 'Consultório 1';
    }
    
    showAddToQueueModal() {
        // Criar modal dinamicamente
        const modal = this.createAddToQueueModal();
        document.body.appendChild(modal);
        
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        // Remover modal quando fechado
        modal.addEventListener('hidden.bs.modal', () => {
            modal.remove();
        });
        
        // Carregar agendamentos de hoje
        this.loadTodayAppointments(modal);
    }
    
    createAddToQueueModal() {
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.innerHTML = `
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Adicionar à Fila de Espera</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Agendamento</label>
                            <select class="form-select" id="select-agendamento">
                                <option value="">Carregando...</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Prioridade</label>
                            <select class="form-select" id="select-prioridade">
                                <option value="normal">Normal</option>
                                <option value="preferencial">Preferencial</option>
                                <option value="urgente">Urgente</option>
                            </select>
                        </div>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            Apenas agendamentos confirmados de hoje podem ser adicionados à fila.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="button" class="btn btn-primary" onclick="queueManager.addToQueue(this)">
                            Adicionar à Fila
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        return modal;
    }
    
    async loadTodayAppointments(modal) {
        try {
            const hoje = new Date().toISOString().split('T')[0];
            const response = await window.app?.apiCall(`agendamentos.php?action=today&data=${hoje}`);
            
            const select = modal.querySelector('#select-agendamento');
            select.innerHTML = '<option value="">Selecione um agendamento</option>';
            
            if (response?.success) {
                const agendamentosDisponiveis = response.data.filter(a => 
                    a.status === 'confirmado' || a.status === 'agendado'
                );
                
                agendamentosDisponiveis.forEach(agendamento => {
                    const option = document.createElement('option');
                    option.value = agendamento.id;
                    option.textContent = `${window.app?.formatTime(agendamento.data_agendamento)} - ${agendamento.paciente_nome} (${agendamento.medico_nome})`;
                    select.appendChild(option);
                });
                
                if (agendamentosDisponiveis.length === 0) {
                    select.innerHTML = '<option value="">Nenhum agendamento disponível</option>';
                }
            }
        } catch (error) {
            console.error('Erro ao carregar agendamentos:', error);
        }
    }
    
    async addToQueue(button) {
        const modal = button.closest('.modal');
        const agendamentoId = modal.querySelector('#select-agendamento').value;
        const prioridade = modal.querySelector('#select-prioridade').value;
        
        if (!agendamentoId) {
            window.app?.showError('Selecione um agendamento');
            return;
        }
        
        try {
            window.app?.showLoading();
            
            const response = await window.app?.apiCall('fila.php?action=add', 'POST', {
                agendamento_id: agendamentoId,
                prioridade: prioridade
            });
            
            if (response?.success) {
                window.app?.showSuccess(`Paciente adicionado à fila com senha ${response.numero_senha}`);
                
                // Fechar modal
                const bsModal = bootstrap.Modal.getInstance(modal);
                bsModal.hide();
                
                await this.loadQueueData();
            } else {
                window.app?.showError(response?.message || 'Erro ao adicionar à fila');
            }
        } catch (error) {
            console.error('Erro ao adicionar à fila:', error);
            window.app?.showError('Erro ao adicionar paciente à fila');
        } finally {
            window.app?.hideLoading();
        }
    }
    
    showPriorityModal(filaId) {
        const modal = this.createPriorityModal(filaId);
        document.body.appendChild(modal);
        
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        modal.addEventListener('hidden.bs.modal', () => {
            modal.remove();
        });
    }
    
    createPriorityModal(filaId) {
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.innerHTML = `
            <div class="modal-dialog modal-sm">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Alterar Prioridade</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Nova Prioridade</label>
                            <select class="form-select" id="new-priority">
                                <option value="normal">Normal</option>
                                <option value="preferencial">Preferencial</option>
                                <option value="urgente">Urgente</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="button" class="btn btn-primary" onclick="queueManager.updatePriority(${filaId}, this)">
                            Atualizar
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        return modal;
    }
    
    async updatePriority(filaId, button) {
        const modal = button.closest('.modal');
        const prioridade = modal.querySelector('#new-priority').value;
        
        try {
            const response = await window.app?.apiCall('fila.php?action=update-priority', 'PUT', {
                fila_id: filaId,
                prioridade: prioridade
            });
            
            if (response?.success) {
                window.app?.showSuccess('Prioridade atualizada');
                
                const bsModal = bootstrap.Modal.getInstance(modal);
                bsModal.hide();
                
                await this.loadQueueData();
            } else {
                window.app?.showError(response?.message || 'Erro ao atualizar prioridade');
            }
        } catch (error) {
            console.error('Erro ao atualizar prioridade:', error);
            window.app?.showError('Erro ao atualizar prioridade');
        }
    }
    
    startPolling() {
        if (this.isPolling) return;
        
        this.isPolling = true;
        this.pollingTimer = setInterval(() => {
            this.loadQueueData();
        }, this.pollingInterval);
    }
    
    stopPolling() {
        if (this.pollingTimer) {
            clearInterval(this.pollingTimer);
            this.pollingTimer = null;
        }
        this.isPolling = false;
    }
    
    playCallSound() {
        try {
            // Criar áudio de chamada
            const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blmGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
            audio.volume = 0.3;
            audio.play().catch(e => console.log('Não foi possível reproduzir som:', e));
        } catch (error) {
            console.log('Áudio não suportado:', error);
        }
    }
    
    // Utility methods
    formatTime(datetime) {
        return new Date(datetime).toLocaleTimeString('pt-BR', {
            hour: '2-digit',
            minute: '2-digit'
        });
    }
    
    formatStatus(status) {
        const statusMap = {
            'aguardando': 'Aguardando',
            'chamado': 'Chamado',
            'atendendo': 'Em Atendimento',
            'finalizado': 'Finalizado'
        };
        return statusMap[status] || status;
    }
    
    formatPriority(priority) {
        const priorityMap = {
            'normal': 'Normal',
            'preferencial': 'Preferencial',
            'urgente': 'Urgente'
        };
        return priorityMap[priority] || priority;
    }
    
    getWaitTimeBadgeClass(minutes) {
        if (minutes < 15) return 'bg-success';
        if (minutes < 30) return 'bg-warning';
        if (minutes < 60) return 'bg-danger';
        return 'bg-dark';
    }
}

// Inicializar gerenciador de fila
document.addEventListener('DOMContentLoaded', () => {
    window.queueManager = new QueueManager();
});

// Extensões para o app principal
if (window.app) {
    // Adicionar método para chamar próximo paciente
    window.app.chamarProximoPaciente = async function() {
        if (window.queueManager) {
            await window.queueManager.callNext();
        }
    };
    
    // Adicionar método para carregar dados da fila
    window.app.loadFilaEspera = async function() {
        if (window.queueManager) {
            await window.queueManager.loadQueueData();
            await window.queueManager.loadQueueStats();
        }
    };
    
    // Iniciar/parar polling baseado na seção atual
    const originalShowSection = window.app.showSection;
    window.app.showSection = function(sectionName) {
        // Parar polling anterior
        if (window.queueManager) {
            window.queueManager.stopPolling();
        }
        
        // Chamar método original
        originalShowSection.call(this, sectionName);
        
        // Iniciar polling se necessário
        if (sectionName === 'fila' && window.queueManager) {
            window.queueManager.startPolling();
        }
    };
}

