/**
 * CliniSys - JavaScript da Página de Login
 * Gerencia autenticação e interações da página de login
 */

class LoginManager {
    constructor() {
        this.apiBase = 'api/';
        this.isLoading = false;
        this.init();
    }
    
    init() {
        this.setupEventHandlers();
        this.setupFormValidation();
        this.addAnimations();
        
        // Verificar se já está logado
        this.checkExistingSession();
    }
    
    setupEventHandlers() {
        // Formulário de login
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleLogin();
        });
        
        // Validação em tempo real
        document.getElementById('email').addEventListener('input', () => {
            this.validateEmail();
        });
        
        document.getElementById('password').addEventListener('input', () => {
            this.validatePassword();
        });
        
        // Teclas de atalho
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' && !this.isLoading) {
                const form = document.getElementById('loginForm');
                if (form.checkValidity()) {
                    this.handleLogin();
                }
            }
        });
    }
    
    setupFormValidation() {
        const emailInput = document.getElementById('email');
        const passwordInput = document.getElementById('password');
        
        // Configurar validação personalizada
        emailInput.addEventListener('blur', () => this.validateEmail());
        passwordInput.addEventListener('blur', () => this.validatePassword());
    }
    
    addAnimations() {
        // Adicionar classes de animação
        document.querySelector('.branding-content')?.classList.add('slide-in-left');
        document.querySelector('.login-form-container')?.classList.add('slide-in-right');
        
        // Animação sequencial dos elementos
        const features = document.querySelectorAll('.feature-item');
        features.forEach((feature, index) => {
            feature.style.animationDelay = `${0.2 + (index * 0.1)}s`;
            feature.classList.add('fade-in');
        });
    }
    
    async checkExistingSession() {
        try {
            const response = await this.apiCall('auth.php?action=me');
            if (response.success) {
                // Usuário já está logado, redirecionar
                window.location.href = 'index.html';
            }
        } catch (error) {
            // Não há sessão ativa, continuar na página de login
            console.log('Nenhuma sessão ativa');
        }
    }
    
    async handleLogin() {
        if (this.isLoading) return;
        
        const email = document.getElementById('email').value.trim();
        const password = document.getElementById('password').value;
        const remember = document.getElementById('remember').checked;
        
        // Validar campos
        if (!this.validateForm()) {
            return;
        }
        
        this.setLoading(true);
        
        try {
            const response = await this.apiCall('auth.php?action=login', 'POST', {
                email: email,
                password: password,
                remember: remember
            });
            
            if (response.success) {
                this.showSuccess('Login realizado com sucesso!');
                
                // Aguardar um pouco antes de redirecionar
                setTimeout(() => {
                    window.location.href = 'index.html';
                }, 1000);
            } else {
                this.showError(response.message || 'Credenciais inválidas');
                this.shakeForm();
            }
        } catch (error) {
            console.error('Erro no login:', error);
            this.showError('Erro de conexão. Tente novamente.');
            this.shakeForm();
        } finally {
            this.setLoading(false);
        }
    }
    
    validateForm() {
        const emailValid = this.validateEmail();
        const passwordValid = this.validatePassword();
        
        return emailValid && passwordValid;
    }
    
    validateEmail() {
        const emailInput = document.getElementById('email');
        const email = emailInput.value.trim();
        const feedback = emailInput.parentNode.querySelector('.form-feedback');
        
        if (!email) {
            this.setFieldState(emailInput, 'invalid', 'E-mail é obrigatório');
            return false;
        }
        
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
            this.setFieldState(emailInput, 'invalid', 'E-mail inválido');
            return false;
        }
        
        this.setFieldState(emailInput, 'valid', 'E-mail válido');
        return true;
    }
    
    validatePassword() {
        const passwordInput = document.getElementById('password');
        const password = passwordInput.value;
        const feedback = passwordInput.parentNode.querySelector('.form-feedback');
        
        if (!password) {
            this.setFieldState(passwordInput, 'invalid', 'Senha é obrigatória');
            return false;
        }
        
        if (password.length < 6) {
            this.setFieldState(passwordInput, 'invalid', 'Senha deve ter pelo menos 6 caracteres');
            return false;
        }
        
        this.setFieldState(passwordInput, 'valid', '');
        return true;
    }
    
    setFieldState(input, state, message) {
        const feedback = input.parentNode.querySelector('.form-feedback');
        
        // Remover classes anteriores
        input.classList.remove('is-valid', 'is-invalid');
        feedback.classList.remove('valid-feedback', 'invalid-feedback');
        
        // Adicionar nova classe
        if (state === 'valid') {
            input.classList.add('is-valid');
            feedback.classList.add('valid-feedback');
        } else if (state === 'invalid') {
            input.classList.add('is-invalid');
            feedback.classList.add('invalid-feedback');
        }
        
        feedback.textContent = message;
    }
    
    setLoading(loading) {
        this.isLoading = loading;
        const button = document.querySelector('.btn-login');
        const btnText = button.querySelector('.btn-text');
        const btnLoading = button.querySelector('.btn-loading');
        
        if (loading) {
            button.disabled = true;
            btnText.classList.add('d-none');
            btnLoading.classList.remove('d-none');
            this.showLoadingOverlay();
        } else {
            button.disabled = false;
            btnText.classList.remove('d-none');
            btnLoading.classList.add('d-none');
            this.hideLoadingOverlay();
        }
    }
    
    showLoadingOverlay() {
        document.getElementById('loadingOverlay').classList.remove('d-none');
    }
    
    hideLoadingOverlay() {
        document.getElementById('loadingOverlay').classList.add('d-none');
    }
    
    shakeForm() {
        const form = document.querySelector('.login-form-container');
        form.style.animation = 'shake 0.5s ease-in-out';
        
        setTimeout(() => {
            form.style.animation = '';
        }, 500);
    }
    
    showSuccess(message) {
        this.showToast(message, 'success');
    }
    
    showError(message) {
        this.showToast(message, 'danger');
    }
    
    showToast(message, type = 'info') {
        // Criar toast dinamicamente
        const toast = document.createElement('div');
        toast.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        toast.style.cssText = `
            top: 20px;
            right: 20px;
            z-index: 10000;
            min-width: 300px;
            animation: slideInRight 0.3s ease-out;
        `;
        
        toast.innerHTML = `
            ${message}
            <button type="button" class="btn-close" onclick="this.parentElement.remove()"></button>
        `;
        
        document.body.appendChild(toast);
        
        // Remover automaticamente após 5 segundos
        setTimeout(() => {
            if (toast.parentNode) {
                toast.style.animation = 'slideOutRight 0.3s ease-in';
                setTimeout(() => toast.remove(), 300);
            }
        }, 5000);
    }
    
    async apiCall(endpoint, method = 'GET', data = null) {
        const url = this.apiBase + endpoint;
        const options = {
            method,
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };
        
        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }
        
        const response = await fetch(url, options);
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        return await response.json();
    }
}

// Funções globais para interações da página
function togglePassword() {
    const passwordInput = document.getElementById('password');
    const toggleButton = document.querySelector('.password-toggle i');
    
    if (passwordInput.type === 'password') {
        passwordInput.type = 'text';
        toggleButton.className = 'fas fa-eye-slash';
    } else {
        passwordInput.type = 'password';
        toggleButton.className = 'fas fa-eye';
    }
}

function fillDemoCredentials(userType) {
    const credentials = {
        admin: {
            email: 'admin@clinisys.com',
            password: 'admin123'
        },
        recepcao: {
            email: 'recepcao@clinisys.com',
            password: 'recepcao123'
        },
        medico: {
            email: 'medico@clinisys.com',
            password: 'medico123'
        }
    };
    
    const cred = credentials[userType];
    if (cred) {
        document.getElementById('email').value = cred.email;
        document.getElementById('password').value = cred.password;
        
        // Validar campos preenchidos
        window.loginManager.validateEmail();
        window.loginManager.validatePassword();
        
        // Feedback visual
        const demoUser = event.target;
        demoUser.style.background = '#28a745';
        demoUser.style.color = 'white';
        
        setTimeout(() => {
            demoUser.style.background = '';
            demoUser.style.color = '';
        }, 1000);
    }
}

function showForgotPassword() {
    // Implementar modal de recuperação de senha
    const modal = createForgotPasswordModal();
    document.body.appendChild(modal);
    
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    modal.addEventListener('hidden.bs.modal', () => {
        modal.remove();
    });
}

function createForgotPasswordModal() {
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.innerHTML = `
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Recuperar Senha</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Digite seu e-mail para receber instruções de recuperação de senha:</p>
                    <div class="mb-3">
                        <label class="form-label">E-mail</label>
                        <input type="email" class="form-control" id="forgot-email" placeholder="seu@email.com">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="button" class="btn btn-primary" onclick="sendPasswordReset()">Enviar</button>
                </div>
            </div>
        </div>
    `;
    
    return modal;
}

function sendPasswordReset() {
    const email = document.getElementById('forgot-email').value;
    
    if (!email) {
        alert('Digite um e-mail válido');
        return;
    }
    
    // Simular envio
    alert('Instruções de recuperação enviadas para ' + email);
    
    // Fechar modal
    const modal = document.querySelector('.modal.show');
    const bsModal = bootstrap.Modal.getInstance(modal);
    bsModal.hide();
}

function showRegister() {
    alert('Funcionalidade de cadastro será implementada em breve!');
}

// Adicionar animação de shake ao CSS dinamicamente
const shakeCSS = `
@keyframes shake {
    0%, 100% { transform: translateX(0); }
    10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
    20%, 40%, 60%, 80% { transform: translateX(5px); }
}

@keyframes slideOutRight {
    to {
        opacity: 0;
        transform: translateX(100%);
    }
}
`;

const style = document.createElement('style');
style.textContent = shakeCSS;
document.head.appendChild(style);

// Inicializar quando DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    window.loginManager = new LoginManager();
});

// Adicionar efeitos de partículas (opcional)
function createParticles() {
    const particlesContainer = document.createElement('div');
    particlesContainer.className = 'particles-container';
    particlesContainer.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        pointer-events: none;
        z-index: -1;
    `;
    
    for (let i = 0; i < 20; i++) {
        const particle = document.createElement('div');
        particle.style.cssText = `
            position: absolute;
            width: 4px;
            height: 4px;
            background: rgba(255, 255, 255, 0.5);
            border-radius: 50%;
            animation: float ${3 + Math.random() * 4}s ease-in-out infinite;
            animation-delay: ${Math.random() * 2}s;
            left: ${Math.random() * 100}%;
            top: ${Math.random() * 100}%;
        `;
        
        particlesContainer.appendChild(particle);
    }
    
    document.querySelector('.login-background').appendChild(particlesContainer);
}

// Criar partículas após carregamento
setTimeout(createParticles, 1000);

