/**
 * CliniSys - Módulo do Calendário
 * Integração com FullCalendar.js
 */

class CalendarManager {
    constructor() {
        this.calendar = null;
        this.currentView = 'dayGridMonth';
        this.init();
    }
    
    init() {
        this.initializeCalendar();
        this.setupEventHandlers();
    }
    
    initializeCalendar() {
        const calendarEl = document.getElementById('calendar');
        if (!calendarEl) return;
        
        this.calendar = new FullCalendar.Calendar(calendarEl, {
            // Configurações básicas
            locale: 'pt-br',
            initialView: this.currentView,
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,timeGridDay'
            },
            
            // Configurações de horário
            slotMinTime: '07:00:00',
            slotMaxTime: '19:00:00',
            slotDuration: '00:30:00',
            slotLabelInterval: '01:00:00',
            
            // Configurações de exibição
            height: 'auto',
            aspectRatio: 1.8,
            nowIndicator: true,
            weekNumbers: false,
            weekends: true,
            
            // Configurações de navegação
            navLinks: true,
            selectable: true,
            selectMirror: true,
            
            // Configurações de eventos
            eventDisplay: 'block',
            eventTimeFormat: {
                hour: '2-digit',
                minute: '2-digit',
                hour12: false
            },
            
            // Configurações de negócios
            businessHours: {
                daysOfWeek: [1, 2, 3, 4, 5, 6], // Segunda a sábado
                startTime: '07:00',
                endTime: '18:00'
            },
            
            // Fonte de eventos
            events: (info, successCallback, failureCallback) => {
                this.loadEvents(info, successCallback, failureCallback);
            },
            
            // Event handlers
            eventClick: (info) => {
                this.handleEventClick(info);
            },
            
            select: (info) => {
                this.handleDateSelect(info);
            },
            
            eventDrop: (info) => {
                this.handleEventDrop(info);
            },
            
            eventResize: (info) => {
                this.handleEventResize(info);
            },
            
            dateClick: (info) => {
                this.handleDateClick(info);
            },
            
            // Customização visual
            eventDidMount: (info) => {
                this.customizeEvent(info);
            },
            
            // Loading
            loading: (isLoading) => {
                if (isLoading) {
                    this.showCalendarLoading();
                } else {
                    this.hideCalendarLoading();
                }
            }
        });
        
        this.calendar.render();
        window.calendar = this.calendar; // Disponibilizar globalmente
    }
    
    setupEventHandlers() {
        // Filtros
        document.getElementById('filtro-medico')?.addEventListener('change', () => {
            this.calendar.refetchEvents();
        });
        
        document.getElementById('filtro-status')?.addEventListener('change', () => {
            this.calendar.refetchEvents();
        });
        
        // Botões de visualização
        document.getElementById('view-month')?.addEventListener('click', () => {
            this.calendar.changeView('dayGridMonth');
            this.updateViewButtons('month');
        });
        
        document.getElementById('view-week')?.addEventListener('click', () => {
            this.calendar.changeView('timeGridWeek');
            this.updateViewButtons('week');
        });
        
        document.getElementById('view-day')?.addEventListener('click', () => {
            this.calendar.changeView('timeGridDay');
            this.updateViewButtons('day');
        });
    }
    
    async loadEvents(info, successCallback, failureCallback) {
        try {
            const params = new URLSearchParams({
                start: info.startStr,
                end: info.endStr
            });
            
            // Adicionar filtros
            const medicoId = document.getElementById('filtro-medico')?.value;
            if (medicoId) {
                params.append('medico_id', medicoId);
            }
            
            const status = document.getElementById('filtro-status')?.value;
            if (status) {
                params.append('status', status);
            }
            
            const response = await fetch(`api/agendamentos.php?action=calendar&${params}`);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            
            const events = await response.json();
            successCallback(events);
            
        } catch (error) {
            console.error('Erro ao carregar eventos:', error);
            failureCallback(error);
            window.app?.showError('Erro ao carregar agendamentos');
        }
    }
    
    handleEventClick(info) {
        const event = info.event;
        const agendamentoId = event.id;
        
        // Mostrar detalhes do agendamento
        this.showEventDetails(agendamentoId, event);
    }
    
    handleDateSelect(info) {
        // Abrir modal de novo agendamento com data pré-selecionada
        const dataAgendamento = document.getElementById('agendamento-data');
        if (dataAgendamento) {
            dataAgendamento.value = info.startStr.split('T')[0];
        }
        
        window.app?.showModal('modalNovoAgendamento');
        
        // Limpar seleção
        this.calendar.unselect();
    }
    
    async handleEventDrop(info) {
        const event = info.event;
        const agendamentoId = event.id;
        const novaDataHora = event.start.toISOString();
        
        try {
            const response = await window.app?.apiCall('agendamentos.php?action=reschedule', 'POST', {
                id: agendamentoId,
                nova_data_hora: novaDataHora
            });
            
            if (response?.success) {
                window.app?.showSuccess('Agendamento reagendado com sucesso');
            } else {
                // Reverter mudança
                info.revert();
                window.app?.showError(response?.message || 'Erro ao reagendar');
            }
        } catch (error) {
            // Reverter mudança
            info.revert();
            window.app?.showError('Erro ao reagendar agendamento');
        }
    }
    
    async handleEventResize(info) {
        const event = info.event;
        const agendamentoId = event.id;
        const inicio = event.start;
        const fim = event.end;
        const duracao = Math.round((fim - inicio) / (1000 * 60)); // em minutos
        
        try {
            const response = await window.app?.apiCall('agendamentos.php?action=update', 'PUT', {
                id: agendamentoId,
                duracao: duracao
            });
            
            if (response?.success) {
                window.app?.showSuccess('Duração atualizada com sucesso');
            } else {
                // Reverter mudança
                info.revert();
                window.app?.showError(response?.message || 'Erro ao atualizar duração');
            }
        } catch (error) {
            // Reverter mudança
            info.revert();
            window.app?.showError('Erro ao atualizar duração');
        }
    }
    
    handleDateClick(info) {
        // Se for visualização de dia, não fazer nada
        if (this.calendar.view.type === 'timeGridDay') {
            return;
        }
        
        // Navegar para visualização de dia
        this.calendar.changeView('timeGridDay', info.date);
        this.updateViewButtons('day');
    }
    
    customizeEvent(info) {
        const event = info.event;
        const element = info.el;
        
        // Adicionar tooltip
        element.title = this.createEventTooltip(event);
        
        // Adicionar classes CSS baseadas no status
        const status = event.extendedProps.status;
        element.classList.add(`event-status-${status}`);
        
        // Adicionar ícone baseado no tipo
        const tipo = event.extendedProps.tipo;
        const icon = this.getTypeIcon(tipo);
        
        if (icon) {
            const iconElement = document.createElement('i');
            iconElement.className = `fas ${icon} me-1`;
            element.querySelector('.fc-event-title')?.prepend(iconElement);
        }
        
        // Adicionar indicador de prioridade se necessário
        if (event.extendedProps.prioridade === 'urgente') {
            element.classList.add('event-urgent');
        }
    }
    
    createEventTooltip(event) {
        const props = event.extendedProps;
        return `
            Paciente: ${props.paciente}
            Médico: ${props.medico}
            Especialidade: ${props.especialidade || 'N/A'}
            Tipo: ${props.tipo}
            Status: ${props.status}
            Horário: ${event.start.toLocaleTimeString('pt-BR', {hour: '2-digit', minute: '2-digit'})}
        `.trim();
    }
    
    getTypeIcon(tipo) {
        const icons = {
            'consulta': 'fa-user-md',
            'retorno': 'fa-redo',
            'exame': 'fa-microscope'
        };
        return icons[tipo] || 'fa-calendar';
    }
    
    showEventDetails(agendamentoId, event) {
        // Criar modal de detalhes dinamicamente
        const modal = this.createEventDetailsModal(agendamentoId, event);
        document.body.appendChild(modal);
        
        const bsModal = new bootstrap.Modal(modal);
        bsModal.show();
        
        // Remover modal quando fechado
        modal.addEventListener('hidden.bs.modal', () => {
            modal.remove();
        });
    }
    
    createEventDetailsModal(agendamentoId, event) {
        const props = event.extendedProps;
        
        const modal = document.createElement('div');
        modal.className = 'modal fade';
        modal.innerHTML = `
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Detalhes do Agendamento</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Paciente:</strong><br>
                                ${props.paciente}
                            </div>
                            <div class="col-md-6">
                                <strong>Médico:</strong><br>
                                ${props.medico}
                            </div>
                        </div>
                        <hr>
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Data/Hora:</strong><br>
                                ${event.start.toLocaleString('pt-BR')}
                            </div>
                            <div class="col-md-6">
                                <strong>Tipo:</strong><br>
                                ${props.tipo}
                            </div>
                        </div>
                        <hr>
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Status:</strong><br>
                                <span class="badge status-${props.status}">${window.app?.formatStatus(props.status)}</span>
                            </div>
                            <div class="col-md-6">
                                <strong>Especialidade:</strong><br>
                                ${props.especialidade || 'N/A'}
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                        <button type="button" class="btn btn-primary" onclick="window.app?.editAgendamento(${agendamentoId})">
                            <i class="fas fa-edit me-1"></i>Editar
                        </button>
                        <button type="button" class="btn btn-success" onclick="window.app?.confirmarAgendamento(${agendamentoId})">
                            <i class="fas fa-check me-1"></i>Confirmar
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        return modal;
    }
    
    updateViewButtons(activeView) {
        // Remover classe active de todos os botões
        document.querySelectorAll('#view-month, #view-week, #view-day').forEach(btn => {
            btn.classList.remove('btn-primary');
            btn.classList.add('btn-outline-primary');
        });
        
        // Adicionar classe active ao botão selecionado
        const activeButton = document.getElementById(`view-${activeView}`);
        if (activeButton) {
            activeButton.classList.remove('btn-outline-primary');
            activeButton.classList.add('btn-primary');
        }
    }
    
    showCalendarLoading() {
        const calendarEl = document.getElementById('calendar');
        if (calendarEl) {
            calendarEl.style.opacity = '0.5';
            calendarEl.style.pointerEvents = 'none';
        }
    }
    
    hideCalendarLoading() {
        const calendarEl = document.getElementById('calendar');
        if (calendarEl) {
            calendarEl.style.opacity = '1';
            calendarEl.style.pointerEvents = 'auto';
        }
    }
    
    // Métodos públicos para controle externo
    refetchEvents() {
        if (this.calendar) {
            this.calendar.refetchEvents();
        }
    }
    
    gotoDate(date) {
        if (this.calendar) {
            this.calendar.gotoDate(date);
        }
    }
    
    changeView(viewName) {
        if (this.calendar) {
            this.calendar.changeView(viewName);
        }
    }
    
    addEvent(eventData) {
        if (this.calendar) {
            this.calendar.addEvent(eventData);
        }
    }
    
    removeEvent(eventId) {
        if (this.calendar) {
            const event = this.calendar.getEventById(eventId);
            if (event) {
                event.remove();
            }
        }
    }
    
    updateEvent(eventId, eventData) {
        if (this.calendar) {
            const event = this.calendar.getEventById(eventId);
            if (event) {
                event.setProp('title', eventData.title);
                event.setStart(eventData.start);
                event.setEnd(eventData.end);
                event.setExtendedProp('status', eventData.status);
            }
        }
    }
}

// Inicializar calendário quando a seção de agendamentos for mostrada
document.addEventListener('DOMContentLoaded', () => {
    // Aguardar um pouco para garantir que o DOM esteja completamente carregado
    setTimeout(() => {
        if (document.getElementById('calendar')) {
            window.calendarManager = new CalendarManager();
        }
    }, 100);
});

// Extensões para o app principal
if (window.app) {
    // Adicionar métodos relacionados ao calendário
    window.app.editAgendamento = async function(agendamentoId) {
        try {
            const response = await this.apiCall(`agendamentos.php?action=show&id=${agendamentoId}`);
            if (response.success) {
                this.populateAgendamentoForm(response.data);
                this.showModal('modalNovoAgendamento');
            }
        } catch (error) {
            this.showError('Erro ao carregar agendamento');
        }
    };
    
    window.app.confirmarAgendamento = async function(agendamentoId) {
        try {
            const response = await this.apiCall('agendamentos.php?action=confirm', 'POST', {id: agendamentoId});
            if (response.success) {
                this.showSuccess('Agendamento confirmado');
                if (window.calendar) {
                    window.calendar.refetchEvents();
                }
            } else {
                this.showError(response.message || 'Erro ao confirmar agendamento');
            }
        } catch (error) {
            this.showError('Erro ao confirmar agendamento');
        }
    };
    
    window.app.populateAgendamentoForm = function(agendamento) {
        document.getElementById('agendamento-paciente').value = agendamento.paciente_id;
        document.getElementById('agendamento-medico').value = agendamento.medico_id;
        document.getElementById('agendamento-data').value = agendamento.data_agendamento.split(' ')[0];
        document.getElementById('agendamento-duracao').value = agendamento.duracao;
        document.getElementById('agendamento-tipo').value = agendamento.tipo_consulta;
        document.getElementById('agendamento-valor').value = agendamento.valor;
        document.getElementById('agendamento-observacoes').value = agendamento.observacoes;
        
        // Carregar horários disponíveis
        this.loadHorariosDisponiveis().then(() => {
            document.getElementById('agendamento-horario').value = agendamento.data_agendamento;
        });
    };
}

