/**
 * CliniSys - JavaScript Principal
 * Gerencia a aplicação e integração com APIs
 */

class CliniSys {
    constructor() {
        this.currentUser = null;
        this.currentSection = 'dashboard';
        this.apiBase = 'api/';
        this.pollingInterval = 5000; // 5 segundos
        this.pollingTimers = {};
        
        this.init();
    }
    
    async init() {
        try {
            // Verificar autenticação
            await this.checkAuth();
            
            // Configurar eventos
            this.setupEvents();
            
            // Carregar dados iniciais
            await this.loadInitialData();
            
            // Iniciar polling
            this.startPolling();
            
            console.log('CliniSys inicializado com sucesso');
        } catch (error) {
            console.error('Erro ao inicializar CliniSys:', error);
            this.showError('Erro ao carregar o sistema');
        }
    }
    
    async checkAuth() {
        try {
            const response = await this.apiCall('auth.php?action=me');
            if (response.success) {
                this.currentUser = response.user;
                this.updateUserInfo();
            } else {
                window.location.href = 'login.html';
            }
        } catch (error) {
            window.location.href = 'login.html';
        }
    }
    
    updateUserInfo() {
        if (this.currentUser) {
            document.getElementById('userName').textContent = this.currentUser.nome;
        }
    }
    
    setupEvents() {
        // Navegação
        document.querySelectorAll('[data-section]').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                this.showSection(link.dataset.section);
            });
        });
        
        // Ações do usuário
        document.querySelectorAll('[data-action]').forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                this.handleAction(button.dataset.action);
            });
        });
        
        // Formulários
        this.setupForms();
        
        // Filtros
        this.setupFilters();
        
        // Teclas de atalho
        this.setupKeyboardShortcuts();
    }
    
    setupForms() {
        // Formulário de novo agendamento
        document.getElementById('btnSalvarAgendamento')?.addEventListener('click', () => {
            this.saveAgendamento();
        });
        
        // Formulário de novo paciente
        document.getElementById('btnSalvarPaciente')?.addEventListener('click', () => {
            this.savePaciente();
        });
        
        // Mudança de médico no agendamento
        document.getElementById('agendamento-medico')?.addEventListener('change', (e) => {
            this.loadHorariosDisponiveis();
        });
        
        // Mudança de data no agendamento
        document.getElementById('agendamento-data')?.addEventListener('change', (e) => {
            this.loadHorariosDisponiveis();
        });
        
        // Busca de pacientes
        document.getElementById('search-pacientes')?.addEventListener('input', 
            this.debounce(() => this.searchPacientes(), 300)
        );
    }
    
    setupFilters() {
        // Filtros do calendário
        document.getElementById('filtro-medico')?.addEventListener('change', () => {
            if (window.calendar) {
                window.calendar.refetchEvents();
            }
        });
        
        document.getElementById('filtro-status')?.addEventListener('change', () => {
            if (window.calendar) {
                window.calendar.refetchEvents();
            }
        });
        
        // Botões de visualização do calendário
        document.getElementById('view-month')?.addEventListener('click', () => {
            if (window.calendar) {
                window.calendar.changeView('dayGridMonth');
            }
        });
        
        document.getElementById('view-week')?.addEventListener('click', () => {
            if (window.calendar) {
                window.calendar.changeView('timeGridWeek');
            }
        });
        
        document.getElementById('view-day')?.addEventListener('click', () => {
            if (window.calendar) {
                window.calendar.changeView('timeGridDay');
            }
        });
    }
    
    setupKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Ctrl/Cmd + N = Novo agendamento
            if ((e.ctrlKey || e.metaKey) && e.key === 'n') {
                e.preventDefault();
                this.handleAction('novo-agendamento');
            }
            
            // Ctrl/Cmd + P = Novo paciente
            if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
                e.preventDefault();
                this.handleAction('novo-paciente');
            }
            
            // F1 = Dashboard
            if (e.key === 'F1') {
                e.preventDefault();
                this.showSection('dashboard');
            }
            
            // F2 = Agendamentos
            if (e.key === 'F2') {
                e.preventDefault();
                this.showSection('agendamentos');
            }
            
            // F3 = Pacientes
            if (e.key === 'F3') {
                e.preventDefault();
                this.showSection('pacientes');
            }
            
            // F4 = Fila
            if (e.key === 'F4') {
                e.preventDefault();
                this.showSection('fila');
            }
        });
    }
    
    async loadInitialData() {
        try {
            // Carregar médicos
            await this.loadMedicos();
            
            // Carregar pacientes para seleção
            await this.loadPacientesSelect();
            
            // Carregar dados do dashboard
            await this.loadDashboardData();
            
        } catch (error) {
            console.error('Erro ao carregar dados iniciais:', error);
        }
    }
    
    async loadMedicos() {
        try {
            const response = await this.apiCall('medicos.php?action=list');
            if (response.success) {
                this.populateSelect('agendamento-medico', response.data, 'id', 'nome');
                this.populateSelect('filtro-medico', response.data, 'id', 'nome');
            }
        } catch (error) {
            console.error('Erro ao carregar médicos:', error);
        }
    }
    
    async loadPacientesSelect() {
        try {
            const response = await this.apiCall('pacientes.php?action=list&per_page=100');
            if (response.success) {
                this.populateSelect('agendamento-paciente', response.data.data, 'id', 'nome');
            }
        } catch (error) {
            console.error('Erro ao carregar pacientes:', error);
        }
    }
    
    async loadDashboardData() {
        try {
            // Carregar estatísticas
            await this.loadDashboardStats();
            
            // Carregar agendamentos de hoje
            await this.loadAgendamentosHoje();
            
        } catch (error) {
            console.error('Erro ao carregar dados do dashboard:', error);
        }
    }
    
    async loadDashboardStats() {
        try {
            const hoje = new Date().toISOString().split('T')[0];
            
            // Agendamentos de hoje
            const agendamentosResponse = await this.apiCall(`agendamentos.php?action=today&data=${hoje}`);
            if (agendamentosResponse.success) {
                const agendamentos = agendamentosResponse.data;
                document.getElementById('stats-agendamentos-hoje').textContent = agendamentos.length;
                
                const finalizados = agendamentos.filter(a => a.status === 'finalizado').length;
                const emAtendimento = agendamentos.filter(a => a.status === 'em_atendimento').length;
                
                document.getElementById('stats-finalizados').textContent = finalizados;
                document.getElementById('stats-atendimentos').textContent = emAtendimento;
            }
            
            // Fila de espera
            const filaResponse = await this.apiCall('fila.php?action=current');
            if (filaResponse.success) {
                const fila = filaResponse.data;
                const aguardando = fila.filter(f => f.status === 'aguardando').length;
                document.getElementById('stats-fila-espera').textContent = aguardando;
            }
            
        } catch (error) {
            console.error('Erro ao carregar estatísticas:', error);
        }
    }
    
    async loadAgendamentosHoje() {
        try {
            const hoje = new Date().toISOString().split('T')[0];
            const response = await this.apiCall(`agendamentos.php?action=today&data=${hoje}`);
            
            if (response.success) {
                this.populateAgendamentosTable(response.data);
            }
        } catch (error) {
            console.error('Erro ao carregar agendamentos de hoje:', error);
        }
    }
    
    populateAgendamentosTable(agendamentos) {
        const tbody = document.querySelector('#agendamentos-hoje-table tbody');
        if (!tbody) return;
        
        tbody.innerHTML = '';
        
        agendamentos.forEach(agendamento => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${this.formatTime(agendamento.data_agendamento)}</td>
                <td>${agendamento.paciente_nome}</td>
                <td>${agendamento.medico_nome}</td>
                <td>${this.formatTipoConsulta(agendamento.tipo_consulta)}</td>
                <td><span class="badge status-${agendamento.status}">${this.formatStatus(agendamento.status)}</span></td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="app.viewAgendamento(${agendamento.id})" title="Visualizar">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-outline-success" onclick="app.confirmarAgendamento(${agendamento.id})" title="Confirmar">
                            <i class="fas fa-check"></i>
                        </button>
                        <button class="btn btn-outline-warning" onclick="app.editAgendamento(${agendamento.id})" title="Editar">
                            <i class="fas fa-edit"></i>
                        </button>
                    </div>
                </td>
            `;
            tbody.appendChild(row);
        });
    }
    
    showSection(sectionName) {
        // Ocultar todas as seções
        document.querySelectorAll('.content-section').forEach(section => {
            section.classList.add('d-none');
        });
        
        // Mostrar seção selecionada
        const targetSection = document.getElementById(`${sectionName}-section`);
        if (targetSection) {
            targetSection.classList.remove('d-none');
            this.currentSection = sectionName;
        }
        
        // Atualizar navegação
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });
        
        document.querySelector(`[data-section="${sectionName}"]`)?.classList.add('active');
        
        // Carregar dados específicos da seção
        this.loadSectionData(sectionName);
    }
    
    async loadSectionData(sectionName) {
        switch (sectionName) {
            case 'dashboard':
                await this.loadDashboardData();
                break;
            case 'agendamentos':
                if (window.calendar) {
                    window.calendar.refetchEvents();
                }
                break;
            case 'pacientes':
                await this.loadPacientes();
                break;
            case 'fila':
                await this.loadFilaEspera();
                break;
            case 'tv':
                await this.loadTVData();
                break;
        }
    }
    
    async handleAction(action) {
        switch (action) {
            case 'novo-agendamento':
                this.showModal('modalNovoAgendamento');
                break;
            case 'novo-paciente':
                this.showModal('modalNovoPaciente');
                break;
            case 'chamar-proximo':
                await this.chamarProximoPaciente();
                break;
            case 'buscar-paciente':
                this.focusSearch();
                break;
            case 'logout':
                await this.logout();
                break;
            case 'profile':
                this.showProfile();
                break;
            case 'change-password':
                this.showChangePassword();
                break;
        }
    }
    
    async saveAgendamento() {
        try {
            const formData = this.getFormData('formNovoAgendamento');
            
            // Validar dados
            if (!this.validateAgendamento(formData)) {
                return;
            }
            
            this.showLoading();
            
            const response = await this.apiCall('agendamentos.php?action=create', 'POST', formData);
            
            if (response.success) {
                this.hideModal('modalNovoAgendamento');
                this.showSuccess('Agendamento criado com sucesso!');
                this.clearForm('formNovoAgendamento');
                
                // Atualizar dados
                if (this.currentSection === 'dashboard') {
                    await this.loadDashboardData();
                }
                if (window.calendar) {
                    window.calendar.refetchEvents();
                }
            } else {
                this.showError(response.message || 'Erro ao criar agendamento');
            }
        } catch (error) {
            this.showError('Erro ao salvar agendamento');
        } finally {
            this.hideLoading();
        }
    }
    
    async savePaciente() {
        try {
            const formData = this.getFormData('formNovoPaciente');
            
            // Validar dados
            if (!this.validatePaciente(formData)) {
                return;
            }
            
            this.showLoading();
            
            const response = await this.apiCall('pacientes.php?action=create', 'POST', formData);
            
            if (response.success) {
                this.hideModal('modalNovoPaciente');
                this.showSuccess('Paciente criado com sucesso!');
                this.clearForm('formNovoPaciente');
                
                // Atualizar listas
                await this.loadPacientesSelect();
                if (this.currentSection === 'pacientes') {
                    await this.loadPacientes();
                }
            } else {
                this.showError(response.message || 'Erro ao criar paciente');
            }
        } catch (error) {
            this.showError('Erro ao salvar paciente');
        } finally {
            this.hideLoading();
        }
    }
    
    async loadHorariosDisponiveis() {
        const medicoId = document.getElementById('agendamento-medico')?.value;
        const data = document.getElementById('agendamento-data')?.value;
        const duracao = document.getElementById('agendamento-duracao')?.value || 30;
        
        const horarioSelect = document.getElementById('agendamento-horario');
        if (!horarioSelect || !medicoId || !data) {
            return;
        }
        
        try {
            const response = await this.apiCall(
                `medicos.php?action=available-times&id=${medicoId}&data=${data}&duracao=${duracao}`
            );
            
            horarioSelect.innerHTML = '<option value="">Selecione um horário</option>';
            
            if (response.success && response.data.length > 0) {
                response.data.forEach(horario => {
                    const option = document.createElement('option');
                    option.value = horario.data_hora;
                    option.textContent = horario.hora;
                    horarioSelect.appendChild(option);
                });
            } else {
                horarioSelect.innerHTML = '<option value="">Nenhum horário disponível</option>';
            }
        } catch (error) {
            console.error('Erro ao carregar horários:', error);
            horarioSelect.innerHTML = '<option value="">Erro ao carregar horários</option>';
        }
    }
    
    startPolling() {
        // Polling para dashboard
        if (this.currentSection === 'dashboard') {
            this.pollingTimers.dashboard = setInterval(() => {
                this.loadDashboardStats();
            }, this.pollingInterval);
        }
        
        // Polling para fila
        if (this.currentSection === 'fila') {
            this.pollingTimers.fila = setInterval(() => {
                this.loadFilaEspera();
            }, this.pollingInterval);
        }
        
        // Polling para TV
        if (this.currentSection === 'tv') {
            this.pollingTimers.tv = setInterval(() => {
                this.loadTVData();
            }, this.pollingInterval);
        }
    }
    
    stopPolling() {
        Object.values(this.pollingTimers).forEach(timer => {
            clearInterval(timer);
        });
        this.pollingTimers = {};
    }
    
    // Utility Methods
    async apiCall(endpoint, method = 'GET', data = null) {
        const url = this.apiBase + endpoint;
        const options = {
            method,
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };
        
        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }
        
        const response = await fetch(url, options);
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        return await response.json();
    }
    
    getFormData(formId) {
        const form = document.getElementById(formId);
        const formData = new FormData(form);
        const data = {};
        
        for (let [key, value] of formData.entries()) {
            data[key] = value;
        }
        
        // Adicionar campos que não estão no FormData
        form.querySelectorAll('input, select, textarea').forEach(field => {
            if (field.name || field.id) {
                const key = field.name || field.id.replace(formId.replace('form', '').toLowerCase() + '-', '');
                data[key] = field.value;
            }
        });
        
        return data;
    }
    
    clearForm(formId) {
        const form = document.getElementById(formId);
        form.reset();
    }
    
    populateSelect(selectId, data, valueField, textField) {
        const select = document.getElementById(selectId);
        if (!select) return;
        
        // Manter primeira opção se existir
        const firstOption = select.querySelector('option');
        select.innerHTML = '';
        
        if (firstOption) {
            select.appendChild(firstOption);
        }
        
        data.forEach(item => {
            const option = document.createElement('option');
            option.value = item[valueField];
            option.textContent = item[textField];
            select.appendChild(option);
        });
    }
    
    showModal(modalId) {
        const modal = new bootstrap.Modal(document.getElementById(modalId));
        modal.show();
    }
    
    hideModal(modalId) {
        const modal = bootstrap.Modal.getInstance(document.getElementById(modalId));
        if (modal) {
            modal.hide();
        }
    }
    
    showLoading() {
        document.getElementById('loadingOverlay')?.classList.remove('d-none');
    }
    
    hideLoading() {
        document.getElementById('loadingOverlay')?.classList.add('d-none');
    }
    
    showSuccess(message) {
        this.showToast(message, 'success');
    }
    
    showError(message) {
        this.showToast(message, 'danger');
    }
    
    showToast(message, type = 'info') {
        // Criar toast dinamicamente
        const toastContainer = document.getElementById('toast-container') || this.createToastContainer();
        
        const toast = document.createElement('div');
        toast.className = `toast align-items-center text-white bg-${type} border-0`;
        toast.setAttribute('role', 'alert');
        toast.innerHTML = `
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        `;
        
        toastContainer.appendChild(toast);
        
        const bsToast = new bootstrap.Toast(toast);
        bsToast.show();
        
        // Remover toast após ser ocultado
        toast.addEventListener('hidden.bs.toast', () => {
            toast.remove();
        });
    }
    
    createToastContainer() {
        const container = document.createElement('div');
        container.id = 'toast-container';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
        return container;
    }
    
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    formatTime(datetime) {
        return new Date(datetime).toLocaleTimeString('pt-BR', {
            hour: '2-digit',
            minute: '2-digit'
        });
    }
    
    formatDate(date) {
        return new Date(date).toLocaleDateString('pt-BR');
    }
    
    formatDateTime(datetime) {
        return new Date(datetime).toLocaleString('pt-BR');
    }
    
    formatStatus(status) {
        const statusMap = {
            'agendado': 'Agendado',
            'confirmado': 'Confirmado',
            'em_atendimento': 'Em Atendimento',
            'finalizado': 'Finalizado',
            'cancelado': 'Cancelado',
            'faltou': 'Faltou'
        };
        return statusMap[status] || status;
    }
    
    formatTipoConsulta(tipo) {
        const tipoMap = {
            'consulta': 'Consulta',
            'retorno': 'Retorno',
            'exame': 'Exame'
        };
        return tipoMap[tipo] || tipo;
    }
    
    validateAgendamento(data) {
        if (!data.paciente_id) {
            this.showError('Selecione um paciente');
            return false;
        }
        
        if (!data.medico_id) {
            this.showError('Selecione um médico');
            return false;
        }
        
        if (!data.data_agendamento) {
            this.showError('Selecione data e horário');
            return false;
        }
        
        return true;
    }
    
    validatePaciente(data) {
        if (!data.nome || data.nome.trim().length < 2) {
            this.showError('Nome deve ter pelo menos 2 caracteres');
            return false;
        }
        
        if (!data.cpf || data.cpf.trim().length < 11) {
            this.showError('CPF é obrigatório');
            return false;
        }
        
        return true;
    }
    
    async logout() {
        try {
            await this.apiCall('auth.php?action=logout', 'POST');
            window.location.href = 'login.html';
        } catch (error) {
            console.error('Erro ao fazer logout:', error);
            window.location.href = 'login.html';
        }
    }
}

// Inicializar aplicação quando DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    window.app = new CliniSys();
});

// Atualizar relógio
function updateClock() {
    const clockElement = document.getElementById('tv-clock');
    if (clockElement) {
        const now = new Date();
        clockElement.textContent = now.toLocaleString('pt-BR');
    }
}

setInterval(updateClock, 1000);
updateClock();

