<?php
/**
 * CliniSys - Configurações do Sistema
 * Versão otimizada para Hostinger e hospedagem compartilhada
 */

// ============================================
// CONFIGURAÇÕES DO BANCO DE DADOS
// ============================================
// IMPORTANTE: Substitua pelos dados do seu banco na Hostinger
define('DB_HOST', 'localhost');
define('DB_NAME', 'u123456789_clinisys');  // Substitua pelo nome do seu banco
define('DB_USER', 'u123456789_clinisys');  // Substitua pelo seu usuário
define('DB_PASS', 'sua_senha_aqui');       // Substitua pela sua senha
define('DB_CHARSET', 'utf8mb4');

// ============================================
// CONFIGURAÇÕES DA APLICAÇÃO
// ============================================
define('APP_NAME', 'CliniSys');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'https://seudominio.com/clinisys'); // Substitua pelo seu domínio

// ============================================
// CONFIGURAÇÕES DE AMBIENTE
// ============================================
// Defina como 'production' quando estiver no ar
define('ENVIRONMENT', 'production'); // 'development' ou 'production'

// ============================================
// CONFIGURAÇÕES DE SESSÃO
// ============================================
define('SESSION_TIMEOUT', 3600); // 1 hora
define('SESSION_NAME', 'clinisys_session');
define('SESSION_SECURE', true);   // true para HTTPS
define('SESSION_HTTPONLY', true); // Segurança adicional

// ============================================
// CONFIGURAÇÕES DE SEGURANÇA
// ============================================
define('HASH_ALGORITHM', 'sha256');
define('SALT', 'seu_salt_unico_aqui_' . md5(__DIR__)); // Gere um salt único
define('CSRF_TOKEN_NAME', 'clinisys_token');

// ============================================
// CONFIGURAÇÕES DE TIMEZONE
// ============================================
date_default_timezone_set('America/Sao_Paulo');

// ============================================
// CONFIGURAÇÕES DE ERRO
// ============================================
if (ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
}

// Definir arquivo de log personalizado
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');

// ============================================
// CONFIGURAÇÕES DA CLÍNICA
// ============================================
define('CLINIC_NAME', 'CliniSys - Clínica Multidisciplinar');
define('CLINIC_ADDRESS', 'Rua das Flores, 123 - Centro - Sua Cidade/SP');
define('CLINIC_PHONE', '(11) 1234-5678');
define('CLINIC_EMAIL', 'contato@suaclinica.com.br');
define('CLINIC_CNPJ', '12.345.678/0001-90');

// ============================================
// CONFIGURAÇÕES DE HORÁRIO
// ============================================
define('CLINIC_OPEN_TIME', '08:00');
define('CLINIC_CLOSE_TIME', '18:00');
define('APPOINTMENT_DURATION', 30); // minutos
define('APPOINTMENT_INTERVAL', 15); // intervalo mínimo entre agendamentos
define('ADVANCE_BOOKING_DAYS', 90); // dias de antecedência para agendamentos

// ============================================
// CONFIGURAÇÕES DA FILA
// ============================================
define('QUEUE_REFRESH_INTERVAL', 5000); // milissegundos
define('QUEUE_CALL_TIMEOUT', 300);      // segundos para repetir chamada
define('QUEUE_MAX_CALLS', 3);           // máximo de chamadas por paciente

// ============================================
// CONFIGURAÇÕES DA TV
// ============================================
define('TV_REFRESH_INTERVAL', 3000);    // milissegundos
define('TV_CALL_DISPLAY_TIME', 30);     // segundos para exibir chamada
define('TV_VOICE_ENABLED', true);       // habilitar voz por padrão

// ============================================
// CONFIGURAÇÕES DE UPLOAD
// ============================================
define('UPLOAD_MAX_SIZE', 5242880);     // 5MB
define('UPLOAD_ALLOWED_TYPES', ['jpg', 'jpeg', 'png', 'pdf', 'doc', 'docx']);
define('UPLOAD_PATH', __DIR__ . '/../uploads/');

// ============================================
// CONFIGURAÇÕES DE E-MAIL (SMTP HOSTINGER)
// ============================================
define('SMTP_ENABLED', false);          // Habilitar quando configurar
define('SMTP_HOST', 'smtp.hostinger.com');
define('SMTP_PORT', 587);
define('SMTP_USER', 'noreply@seudominio.com'); // Substitua
define('SMTP_PASS', 'sua_senha_email');        // Substitua
define('SMTP_SECURE', 'tls');
define('SMTP_FROM_NAME', CLINIC_NAME);

// ============================================
// CONFIGURAÇÕES DE CACHE
// ============================================
define('CACHE_ENABLED', true);
define('CACHE_DURATION', 300); // 5 minutos
define('CACHE_PATH', __DIR__ . '/../temp/cache/');

// ============================================
// CONFIGURAÇÕES DE LOG
// ============================================
define('LOG_ENABLED', true);
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_PATH', __DIR__ . '/../logs/');
define('LOG_MAX_SIZE', 10485760); // 10MB

// ============================================
// CONFIGURAÇÕES DE API
// ============================================
define('API_RATE_LIMIT', 100);          // requests por minuto
define('API_TIMEOUT', 30);              // segundos
define('API_CORS_ENABLED', true);       // Habilitar CORS

// ============================================
// CONFIGURAÇÕES DE BACKUP
// ============================================
define('BACKUP_ENABLED', false);        // Habilitar backup automático
define('BACKUP_PATH', __DIR__ . '/../backups/');
define('BACKUP_RETENTION_DAYS', 30);    // dias para manter backups

// ============================================
// CONFIGURAÇÕES DE MANUTENÇÃO
// ============================================
define('MAINTENANCE_MODE', false);      // Modo manutenção
define('MAINTENANCE_MESSAGE', 'Sistema em manutenção. Tente novamente em alguns minutos.');

// ============================================
// CONFIGURAÇÕES ESPECÍFICAS DA HOSTINGER
// ============================================
// Ajustes para hospedagem compartilhada
ini_set('max_execution_time', 300);
ini_set('memory_limit', '256M');
ini_set('upload_max_filesize', '10M');
ini_set('post_max_size', '10M');

// ============================================
// FUNÇÕES AUXILIARES
// ============================================

/**
 * Obter configuração do banco de dados
 */
function getDbConfig() {
    return [
        'host' => DB_HOST,
        'dbname' => DB_NAME,
        'username' => DB_USER,
        'password' => DB_PASS,
        'charset' => DB_CHARSET,
        'options' => [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
        ]
    ];
}

/**
 * Verificar se está em modo de desenvolvimento
 */
function isDevelopment() {
    return ENVIRONMENT === 'development';
}

/**
 * Verificar se está em modo de produção
 */
function isProduction() {
    return ENVIRONMENT === 'production';
}

/**
 * Obter URL base da aplicação
 */
function getBaseUrl() {
    return APP_URL;
}

/**
 * Obter caminho absoluto
 */
function getAbsolutePath($path = '') {
    return __DIR__ . '/../' . ltrim($path, '/');
}

/**
 * Verificar se modo manutenção está ativo
 */
function isMaintenanceMode() {
    return MAINTENANCE_MODE;
}

/**
 * Log de erro personalizado
 */
function logError($message, $context = []) {
    if (LOG_ENABLED) {
        $logFile = LOG_PATH . 'error_' . date('Y-m-d') . '.log';
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' | Context: ' . json_encode($context) : '';
        $logMessage = "[$timestamp] ERROR: $message$contextStr" . PHP_EOL;
        
        // Criar diretório se não existir
        if (!is_dir(LOG_PATH)) {
            mkdir(LOG_PATH, 0755, true);
        }
        
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

/**
 * Log de informação
 */
function logInfo($message, $context = []) {
    if (LOG_ENABLED && in_array(LOG_LEVEL, ['DEBUG', 'INFO'])) {
        $logFile = LOG_PATH . 'info_' . date('Y-m-d') . '.log';
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' | Context: ' . json_encode($context) : '';
        $logMessage = "[$timestamp] INFO: $message$contextStr" . PHP_EOL;
        
        if (!is_dir(LOG_PATH)) {
            mkdir(LOG_PATH, 0755, true);
        }
        
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

// ============================================
// INICIALIZAÇÃO
// ============================================

// Criar diretórios necessários
$directories = [
    LOG_PATH,
    UPLOAD_PATH,
    CACHE_PATH,
    __DIR__ . '/../temp/'
];

foreach ($directories as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Verificar modo manutenção
if (isMaintenanceMode() && !isset($_GET['admin_access'])) {
    http_response_code(503);
    die('<!DOCTYPE html>
    <html>
    <head>
        <title>Sistema em Manutenção</title>
        <meta charset="UTF-8">
        <style>
            body { font-family: Arial, sans-serif; text-align: center; padding: 50px; }
            .maintenance { background: #f8f9fa; padding: 30px; border-radius: 10px; display: inline-block; }
        </style>
    </head>
    <body>
        <div class="maintenance">
            <h1>🔧 Sistema em Manutenção</h1>
            <p>' . MAINTENANCE_MESSAGE . '</p>
        </div>
    </body>
    </html>');
}

// Log de inicialização
if (LOG_ENABLED) {
    logInfo('Sistema inicializado', [
        'environment' => ENVIRONMENT,
        'php_version' => PHP_VERSION,
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
    ]);
}

?>

