<?php
/**
 * CliniSys - Classe Paciente
 * Gerencia operações relacionadas aos pacientes
 */

require_once 'BaseEntity.php';

class Paciente extends BaseEntity {
    protected $table = 'pacientes';
    protected $fillable = [
        'nome', 'cpf', 'rg', 'data_nascimento', 'telefone', 'celular',
        'email', 'endereco', 'convenio', 'numero_convenio', 'observacoes', 'ativo'
    ];
    
    // Buscar paciente por CPF
    public function findByCpf($cpf) {
        return $this->whereOne('cpf = :cpf', ['cpf' => $cpf]);
    }
    
    // Buscar pacientes por nome (busca parcial)
    public function searchByName($nome) {
        return $this->where('nome LIKE :nome AND ativo = 1', ['nome' => "%{$nome}%"]);
    }
    
    // Buscar pacientes por convênio
    public function findByConvenio($convenio) {
        return $this->where('convenio = :convenio AND ativo = 1', ['convenio' => $convenio]);
    }
    
    // Obter histórico de agendamentos do paciente
    public function getAgendamentos($pacienteId, $limit = 10) {
        $sql = "
            SELECT a.*, m.nome as medico_nome, e.nome as especialidade_nome, u.nome as usuario_nome
            FROM agendamentos a
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE a.paciente_id = :paciente_id
            ORDER BY a.data_agendamento DESC
            LIMIT {$limit}
        ";
        
        return $this->db->select($sql, ['paciente_id' => $pacienteId]);
    }
    
    // Obter próximos agendamentos do paciente
    public function getProximosAgendamentos($pacienteId) {
        $sql = "
            SELECT a.*, m.nome as medico_nome, e.nome as especialidade_nome, u.nome as usuario_nome
            FROM agendamentos a
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE a.paciente_id = :paciente_id 
            AND a.data_agendamento >= NOW()
            AND a.status IN ('agendado', 'confirmado')
            ORDER BY a.data_agendamento ASC
        ";
        
        return $this->db->select($sql, ['paciente_id' => $pacienteId]);
    }
    
    // Validar dados do paciente
    public function validatePaciente($data, $isUpdate = false) {
        $rules = [
            'nome' => 'required|min:2|max:100',
            'telefone' => 'min:10|max:20',
            'celular' => 'min:10|max:20',
            'email' => 'email|max:100'
        ];
        
        if (!$isUpdate) {
            $rules['cpf'] = 'required|min:11|max:14';
        }
        
        $errors = $this->validate($data, $rules);
        
        // Validação específica de CPF
        if (!empty($data['cpf']) && !$this->validarCpf($data['cpf'])) {
            $errors['cpf'] = 'CPF inválido';
        }
        
        // Verificar se CPF já existe (apenas para novos pacientes)
        if (!$isUpdate && !empty($data['cpf'])) {
            $existing = $this->findByCpf($data['cpf']);
            if ($existing) {
                $errors['cpf'] = 'CPF já cadastrado';
            }
        }
        
        return $errors;
    }
    
    // Validar CPF
    private function validarCpf($cpf) {
        // Remove caracteres não numéricos
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        
        // Verifica se tem 11 dígitos
        if (strlen($cpf) != 11) {
            return false;
        }
        
        // Verifica se todos os dígitos são iguais
        if (preg_match('/(\d)\1{10}/', $cpf)) {
            return false;
        }
        
        // Calcula os dígitos verificadores
        for ($t = 9; $t < 11; $t++) {
            for ($d = 0, $c = 0; $c < $t; $c++) {
                $d += $cpf[$c] * (($t + 1) - $c);
            }
            $d = ((10 * $d) % 11) % 10;
            if ($cpf[$c] != $d) {
                return false;
            }
        }
        
        return true;
    }
    
    // Formatar CPF
    public function formatarCpf($cpf) {
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        return preg_replace('/(\d{3})(\d{3})(\d{3})(\d{2})/', '$1.$2.$3-$4', $cpf);
    }
    
    // Formatar telefone
    public function formatarTelefone($telefone) {
        $telefone = preg_replace('/[^0-9]/', '', $telefone);
        
        if (strlen($telefone) == 11) {
            return preg_replace('/(\d{2})(\d{5})(\d{4})/', '($1) $2-$3', $telefone);
        } elseif (strlen($telefone) == 10) {
            return preg_replace('/(\d{2})(\d{4})(\d{4})/', '($1) $2-$3', $telefone);
        }
        
        return $telefone;
    }
    
    // Obter estatísticas do paciente
    public function getEstatisticas($pacienteId) {
        $sql = "
            SELECT 
                COUNT(*) as total_agendamentos,
                COUNT(CASE WHEN status = 'finalizado' THEN 1 END) as consultas_realizadas,
                COUNT(CASE WHEN status = 'cancelado' THEN 1 END) as consultas_canceladas,
                COUNT(CASE WHEN status = 'faltou' THEN 1 END) as faltas,
                MIN(data_agendamento) as primeira_consulta,
                MAX(data_agendamento) as ultima_consulta
            FROM agendamentos 
            WHERE paciente_id = :paciente_id
        ";
        
        return $this->db->selectOne($sql, ['paciente_id' => $pacienteId]);
    }
    
    // Buscar pacientes com aniversário hoje
    public function getAniversariantes() {
        $sql = "
            SELECT * FROM pacientes 
            WHERE DAY(data_nascimento) = DAY(CURDATE()) 
            AND MONTH(data_nascimento) = MONTH(CURDATE())
            AND ativo = 1
            ORDER BY nome
        ";
        
        return $this->db->select($sql);
    }
    
    // Buscar pacientes por idade
    public function findByIdade($idadeMin, $idadeMax) {
        $sql = "
            SELECT *, TIMESTAMPDIFF(YEAR, data_nascimento, CURDATE()) as idade
            FROM pacientes 
            WHERE TIMESTAMPDIFF(YEAR, data_nascimento, CURDATE()) BETWEEN :idade_min AND :idade_max
            AND ativo = 1
            ORDER BY nome
        ";
        
        return $this->db->select($sql, [
            'idade_min' => $idadeMin,
            'idade_max' => $idadeMax
        ]);
    }
}
?>

