<?php
/**
 * CliniSys - Classe Medico
 * Gerencia operações relacionadas aos médicos
 */

require_once 'BaseEntity.php';

class Medico extends BaseEntity {
    protected $table = 'medicos';
    protected $fillable = ['usuario_id', 'crm', 'especialidade_id', 'telefone', 'ativo'];
    
    // Buscar todos os médicos com informações completas
    public function getAllWithDetails() {
        $sql = "
            SELECT m.*, u.nome, u.email, e.nome as especialidade_nome, e.cor as especialidade_cor
            FROM medicos m
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE m.ativo = 1 AND u.ativo = 1
            ORDER BY u.nome
        ";
        
        return $this->db->select($sql);
    }
    
    // Buscar médico por CRM
    public function findByCrm($crm) {
        $sql = "
            SELECT m.*, u.nome, u.email, e.nome as especialidade_nome
            FROM medicos m
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE m.crm = :crm AND m.ativo = 1
        ";
        
        return $this->db->selectOne($sql, ['crm' => $crm]);
    }
    
    // Buscar médicos por especialidade
    public function findByEspecialidade($especialidadeId) {
        $sql = "
            SELECT m.*, u.nome, u.email, e.nome as especialidade_nome
            FROM medicos m
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE m.especialidade_id = :especialidade_id AND m.ativo = 1 AND u.ativo = 1
            ORDER BY u.nome
        ";
        
        return $this->db->select($sql, ['especialidade_id' => $especialidadeId]);
    }
    
    // Obter horários de funcionamento do médico
    public function getHorariosFuncionamento($medicoId) {
        $sql = "
            SELECT * FROM horarios_funcionamento 
            WHERE medico_id = :medico_id AND ativo = 1
            ORDER BY dia_semana, hora_inicio
        ";
        
        return $this->db->select($sql, ['medico_id' => $medicoId]);
    }
    
    // Definir horários de funcionamento
    public function setHorariosFuncionamento($medicoId, $horarios) {
        try {
            $this->db->beginTransaction();
            
            // Remover horários existentes
            $this->db->delete('horarios_funcionamento', 'medico_id = :medico_id', ['medico_id' => $medicoId]);
            
            // Inserir novos horários
            foreach ($horarios as $horario) {
                $this->db->insert('horarios_funcionamento', [
                    'medico_id' => $medicoId,
                    'dia_semana' => $horario['dia_semana'],
                    'hora_inicio' => $horario['hora_inicio'],
                    'hora_fim' => $horario['hora_fim'],
                    'ativo' => 1
                ]);
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            return false;
        }
    }
    
    // Verificar disponibilidade do médico
    public function isDisponivel($medicoId, $dataHora, $duracao = 30) {
        $dataInicio = new DateTime($dataHora);
        $dataFim = clone $dataInicio;
        $dataFim->add(new DateInterval('PT' . $duracao . 'M'));
        
        // Verificar se está dentro do horário de funcionamento
        $diaSemana = $dataInicio->format('w'); // 0 = domingo, 1 = segunda, etc.
        $hora = $dataInicio->format('H:i:s');
        
        $sql = "
            SELECT COUNT(*) as count FROM horarios_funcionamento 
            WHERE medico_id = :medico_id 
            AND dia_semana = :dia_semana 
            AND hora_inicio <= :hora 
            AND hora_fim >= :hora
            AND ativo = 1
        ";
        
        $horarioValido = $this->db->selectOne($sql, [
            'medico_id' => $medicoId,
            'dia_semana' => $diaSemana,
            'hora' => $hora
        ]);
        
        if ($horarioValido['count'] == 0) {
            return false;
        }
        
        // Verificar conflitos com outros agendamentos
        $sql = "
            SELECT COUNT(*) as count FROM agendamentos 
            WHERE medico_id = :medico_id 
            AND status NOT IN ('cancelado', 'faltou')
            AND (
                (data_agendamento <= :data_inicio AND DATE_ADD(data_agendamento, INTERVAL duracao MINUTE) > :data_inicio)
                OR
                (data_agendamento < :data_fim AND data_agendamento >= :data_inicio)
            )
        ";
        
        $conflito = $this->db->selectOne($sql, [
            'medico_id' => $medicoId,
            'data_inicio' => $dataInicio->format('Y-m-d H:i:s'),
            'data_fim' => $dataFim->format('Y-m-d H:i:s')
        ]);
        
        return $conflito['count'] == 0;
    }
    
    // Obter agenda do médico para um período
    public function getAgenda($medicoId, $dataInicio, $dataFim) {
        $sql = "
            SELECT a.*, p.nome as paciente_nome, p.telefone as paciente_telefone,
                   p.celular as paciente_celular, p.convenio as paciente_convenio
            FROM agendamentos a
            INNER JOIN pacientes p ON a.paciente_id = p.id
            WHERE a.medico_id = :medico_id
            AND a.data_agendamento BETWEEN :data_inicio AND :data_fim
            AND a.status NOT IN ('cancelado')
            ORDER BY a.data_agendamento
        ";
        
        return $this->db->select($sql, [
            'medico_id' => $medicoId,
            'data_inicio' => $dataInicio,
            'data_fim' => $dataFim
        ]);
    }
    
    // Obter horários disponíveis para agendamento
    public function getHorariosDisponiveis($medicoId, $data, $duracao = 30) {
        $horariosDisponiveis = [];
        $diaSemana = date('w', strtotime($data));
        
        // Buscar horários de funcionamento para o dia
        $horariosFuncionamento = $this->db->select(
            "SELECT * FROM horarios_funcionamento WHERE medico_id = :medico_id AND dia_semana = :dia_semana AND ativo = 1",
            ['medico_id' => $medicoId, 'dia_semana' => $diaSemana]
        );
        
        foreach ($horariosFuncionamento as $horario) {
            $inicio = new DateTime($data . ' ' . $horario['hora_inicio']);
            $fim = new DateTime($data . ' ' . $horario['hora_fim']);
            
            // Gerar slots de tempo
            while ($inicio < $fim) {
                $proximoSlot = clone $inicio;
                $proximoSlot->add(new DateInterval('PT' . $duracao . 'M'));
                
                if ($proximoSlot <= $fim) {
                    $dataHora = $inicio->format('Y-m-d H:i:s');
                    
                    if ($this->isDisponivel($medicoId, $dataHora, $duracao)) {
                        $horariosDisponiveis[] = [
                            'data_hora' => $dataHora,
                            'hora' => $inicio->format('H:i'),
                            'disponivel' => true
                        ];
                    }
                }
                
                $inicio->add(new DateInterval('PT' . $duracao . 'M'));
            }
        }
        
        return $horariosDisponiveis;
    }
    
    // Obter estatísticas do médico
    public function getEstatisticas($medicoId, $periodo = '30 days') {
        $sql = "
            SELECT 
                COUNT(*) as total_agendamentos,
                COUNT(CASE WHEN status = 'finalizado' THEN 1 END) as consultas_realizadas,
                COUNT(CASE WHEN status = 'cancelado' THEN 1 END) as consultas_canceladas,
                COUNT(CASE WHEN status = 'faltou' THEN 1 END) as faltas,
                AVG(valor) as valor_medio,
                SUM(CASE WHEN status = 'finalizado' THEN valor ELSE 0 END) as faturamento
            FROM agendamentos 
            WHERE medico_id = :medico_id 
            AND data_agendamento >= DATE_SUB(NOW(), INTERVAL {$periodo})
        ";
        
        return $this->db->selectOne($sql, ['medico_id' => $medicoId]);
    }
    
    // Obter próximos agendamentos do médico
    public function getProximosAgendamentos($medicoId, $limit = 5) {
        $sql = "
            SELECT a.*, p.nome as paciente_nome, p.telefone as paciente_telefone
            FROM agendamentos a
            INNER JOIN pacientes p ON a.paciente_id = p.id
            WHERE a.medico_id = :medico_id
            AND a.data_agendamento >= NOW()
            AND a.status IN ('agendado', 'confirmado')
            ORDER BY a.data_agendamento
            LIMIT {$limit}
        ";
        
        return $this->db->select($sql, ['medico_id' => $medicoId]);
    }
    
    // Validar dados do médico
    public function validateMedico($data, $isUpdate = false) {
        $rules = [
            'crm' => 'required|min:4|max:20',
            'telefone' => 'min:10|max:20'
        ];
        
        $errors = $this->validate($data, $rules);
        
        // Verificar se CRM já existe (apenas para novos médicos)
        if (!$isUpdate && !empty($data['crm'])) {
            $existing = $this->findByCrm($data['crm']);
            if ($existing) {
                $errors['crm'] = 'CRM já cadastrado';
            }
        }
        
        return $errors;
    }
}
?>

