<?php
/**
 * CliniSys - Classe Fila de Espera
 * Gerencia operações relacionadas à fila de espera
 */

require_once 'BaseEntity.php';

class FilaEspera extends BaseEntity {
    protected $table = 'fila_espera';
    protected $fillable = [
        'agendamento_id', 'numero_senha', 'data_chegada', 'status', 
        'prioridade', 'chamado_em', 'atendido_em', 'finalizado_em', 'observacoes'
    ];
    
    // Adicionar paciente à fila
    public function adicionarPaciente($agendamentoId, $prioridade = 'normal') {
        try {
            // Verificar se o agendamento existe e está válido
            $agendamento = new Agendamento();
            $dadosAgendamento = $agendamento->find($agendamentoId);
            
            if (!$dadosAgendamento) {
                return ['success' => false, 'message' => 'Agendamento não encontrado'];
            }
            
            if ($dadosAgendamento['status'] !== 'confirmado' && $dadosAgendamento['status'] !== 'agendado') {
                return ['success' => false, 'message' => 'Agendamento não está em status válido'];
            }
            
            // Verificar se já está na fila
            $jaExiste = $this->whereOne('agendamento_id = :agendamento_id AND status != "finalizado"', 
                                      ['agendamento_id' => $agendamentoId]);
            
            if ($jaExiste) {
                return ['success' => false, 'message' => 'Paciente já está na fila'];
            }
            
            // Gerar número da senha
            $numeroSenha = $this->gerarNumeroSenha();
            
            // Adicionar à fila
            $filaId = $this->create([
                'agendamento_id' => $agendamentoId,
                'numero_senha' => $numeroSenha,
                'status' => 'aguardando',
                'prioridade' => $prioridade
            ]);
            
            // Atualizar status do agendamento
            $agendamento->update($agendamentoId, ['status' => 'confirmado']);
            
            return [
                'success' => true,
                'fila_id' => $filaId,
                'numero_senha' => $numeroSenha,
                'message' => 'Paciente adicionado à fila com sucesso'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Erro ao adicionar paciente à fila'];
        }
    }
    
    // Gerar número da senha
    private function gerarNumeroSenha() {
        $hoje = date('Y-m-d');
        
        // Buscar o último número do dia
        $sql = "SELECT MAX(numero_senha) as ultimo_numero 
                FROM fila_espera 
                WHERE DATE(data_chegada) = :hoje";
        
        $resultado = $this->db->selectOne($sql, ['hoje' => $hoje]);
        $ultimoNumero = $resultado['ultimo_numero'] ?? 0;
        
        return $ultimoNumero + 1;
    }
    
    // Obter fila atual com detalhes
    public function getFilaAtual() {
        $sql = "
            SELECT f.*, a.data_agendamento, a.tipo_consulta,
                   p.nome as paciente_nome, p.convenio as paciente_convenio,
                   u.nome as medico_nome, m.crm as medico_crm,
                   e.nome as especialidade_nome, e.cor as especialidade_cor,
                   TIMESTAMPDIFF(MINUTE, f.data_chegada, NOW()) as tempo_espera
            FROM fila_espera f
            INNER JOIN agendamentos a ON f.agendamento_id = a.id
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE f.status IN ('aguardando', 'chamado', 'atendendo')
            AND DATE(f.data_chegada) = CURDATE()
            ORDER BY 
                CASE f.prioridade 
                    WHEN 'urgente' THEN 1 
                    WHEN 'preferencial' THEN 2 
                    ELSE 3 
                END,
                f.data_chegada
        ";
        
        return $this->db->select($sql);
    }
    
    // Chamar próximo paciente
    public function chamarProximo($consultorio = null) {
        try {
            // Buscar próximo paciente na fila
            $sql = "
                SELECT f.*, p.nome as paciente_nome, u.nome as medico_nome
                FROM fila_espera f
                INNER JOIN agendamentos a ON f.agendamento_id = a.id
                INNER JOIN pacientes p ON a.paciente_id = p.id
                INNER JOIN medicos m ON a.medico_id = m.id
                INNER JOIN usuarios u ON m.usuario_id = u.id
                WHERE f.status = 'aguardando'
                AND DATE(f.data_chegada) = CURDATE()
                ORDER BY 
                    CASE f.prioridade 
                        WHEN 'urgente' THEN 1 
                        WHEN 'preferencial' THEN 2 
                        ELSE 3 
                    END,
                    f.data_chegada
                LIMIT 1
            ";
            
            $proximoPaciente = $this->db->selectOne($sql);
            
            if (!$proximoPaciente) {
                return ['success' => false, 'message' => 'Nenhum paciente na fila'];
            }
            
            // Atualizar status para chamado
            $this->update($proximoPaciente['id'], [
                'status' => 'chamado',
                'chamado_em' => date('Y-m-d H:i:s')
            ]);
            
            // Criar chamada para TV
            $this->criarChamadaTV($proximoPaciente['id'], $consultorio);
            
            return [
                'success' => true,
                'paciente' => $proximoPaciente,
                'message' => 'Paciente chamado com sucesso'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Erro ao chamar paciente'];
        }
    }
    
    // Criar chamada para TV
    private function criarChamadaTV($filaId, $consultorio = null) {
        $fila = $this->find($filaId);
        if (!$fila) return false;
        
        // Buscar dados completos
        $sql = "
            SELECT f.numero_senha, p.nome as paciente_nome, u.nome as medico_nome
            FROM fila_espera f
            INNER JOIN agendamentos a ON f.agendamento_id = a.id
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            WHERE f.id = :fila_id
        ";
        
        $dados = $this->db->selectOne($sql, ['fila_id' => $filaId]);
        
        if ($dados) {
            $this->db->insert('chamadas_tv', [
                'fila_id' => $filaId,
                'numero_senha' => $dados['numero_senha'],
                'nome_paciente' => $dados['paciente_nome'],
                'medico' => $dados['medico_nome'],
                'consultorio' => $consultorio ?? 'Consultório 1',
                'tipo_chamada' => 'primeira',
                'exibido' => false
            ]);
        }
    }
    
    // Confirmar atendimento
    public function confirmarAtendimento($filaId) {
        $result = $this->update($filaId, [
            'status' => 'atendendo',
            'atendido_em' => date('Y-m-d H:i:s')
        ]);
        
        if ($result) {
            // Atualizar agendamento
            $fila = $this->find($filaId);
            if ($fila) {
                $agendamento = new Agendamento();
                $agendamento->update($fila['agendamento_id'], ['status' => 'em_atendimento']);
            }
        }
        
        return $result;
    }
    
    // Finalizar atendimento
    public function finalizarAtendimento($filaId) {
        $result = $this->update($filaId, [
            'status' => 'finalizado',
            'finalizado_em' => date('Y-m-d H:i:s')
        ]);
        
        if ($result) {
            // Atualizar agendamento
            $fila = $this->find($filaId);
            if ($fila) {
                $agendamento = new Agendamento();
                $agendamento->update($fila['agendamento_id'], ['status' => 'finalizado']);
            }
        }
        
        return $result;
    }
    
    // Remarcar chamada (segunda ou terceira chamada)
    public function remarcarChamada($filaId, $consultorio = null) {
        $fila = $this->find($filaId);
        if (!$fila || $fila['status'] !== 'chamado') {
            return ['success' => false, 'message' => 'Paciente não está em status de chamado'];
        }
        
        // Verificar quantas chamadas já foram feitas
        $sql = "SELECT COUNT(*) as total FROM chamadas_tv WHERE fila_id = :fila_id";
        $resultado = $this->db->selectOne($sql, ['fila_id' => $filaId]);
        $totalChamadas = $resultado['total'];
        
        if ($totalChamadas >= 3) {
            return ['success' => false, 'message' => 'Limite de chamadas atingido'];
        }
        
        $tipoChamada = $totalChamadas == 1 ? 'segunda' : 'terceira';
        
        // Buscar dados para nova chamada
        $sql = "
            SELECT f.numero_senha, p.nome as paciente_nome, u.nome as medico_nome
            FROM fila_espera f
            INNER JOIN agendamentos a ON f.agendamento_id = a.id
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            WHERE f.id = :fila_id
        ";
        
        $dados = $this->db->selectOne($sql, ['fila_id' => $filaId]);
        
        if ($dados) {
            $this->db->insert('chamadas_tv', [
                'fila_id' => $filaId,
                'numero_senha' => $dados['numero_senha'],
                'nome_paciente' => $dados['paciente_nome'],
                'medico' => $dados['medico_nome'],
                'consultorio' => $consultorio ?? 'Consultório 1',
                'tipo_chamada' => $tipoChamada,
                'exibido' => false
            ]);
            
            return ['success' => true, 'message' => ucfirst($tipoChamada) . ' chamada criada'];
        }
        
        return ['success' => false, 'message' => 'Erro ao criar nova chamada'];
    }
    
    // Remover da fila (desistência)
    public function removerDaFila($filaId, $motivo = 'Desistência') {
        $result = $this->update($filaId, [
            'status' => 'finalizado',
            'finalizado_em' => date('Y-m-d H:i:s'),
            'observacoes' => $motivo
        ]);
        
        if ($result) {
            // Atualizar agendamento como cancelado
            $fila = $this->find($filaId);
            if ($fila) {
                $agendamento = new Agendamento();
                $agendamento->update($fila['agendamento_id'], ['status' => 'cancelado']);
            }
        }
        
        return $result;
    }
    
    // Obter estatísticas da fila
    public function getEstatisticas($data = null) {
        if (!$data) {
            $data = date('Y-m-d');
        }
        
        $sql = "
            SELECT 
                COUNT(*) as total_pacientes,
                COUNT(CASE WHEN status = 'aguardando' THEN 1 END) as aguardando,
                COUNT(CASE WHEN status = 'chamado' THEN 1 END) as chamados,
                COUNT(CASE WHEN status = 'atendendo' THEN 1 END) as atendendo,
                COUNT(CASE WHEN status = 'finalizado' THEN 1 END) as finalizados,
                AVG(CASE WHEN finalizado_em IS NOT NULL 
                    THEN TIMESTAMPDIFF(MINUTE, data_chegada, finalizado_em) 
                    ELSE NULL END) as tempo_medio_espera,
                MAX(CASE WHEN status = 'aguardando' 
                    THEN TIMESTAMPDIFF(MINUTE, data_chegada, NOW()) 
                    ELSE 0 END) as maior_tempo_espera
            FROM fila_espera 
            WHERE DATE(data_chegada) = :data
        ";
        
        return $this->db->selectOne($sql, ['data' => $data]);
    }
    
    // Obter chamadas para TV (não exibidas)
    public function getChamadasTV() {
        $sql = "
            SELECT * FROM chamadas_tv 
            WHERE exibido = 0 
            ORDER BY created_at DESC 
            LIMIT " . MAX_CHAMADAS_EXIBICAO;
        
        return $this->db->select($sql);
    }
    
    // Marcar chamadas como exibidas
    public function marcarChamadasExibidas($ids) {
        if (empty($ids)) return false;
        
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $sql = "UPDATE chamadas_tv SET exibido = 1 WHERE id IN ({$placeholders})";
        
        return $this->db->query($sql, $ids);
    }
    
    // Limpar chamadas antigas
    public function limparChamadasAntigas($dias = 7) {
        $sql = "DELETE FROM chamadas_tv WHERE created_at < DATE_SUB(NOW(), INTERVAL :dias DAY)";
        return $this->db->query($sql, ['dias' => $dias]);
    }
}
?>

