<?php
/**
 * CliniSys - Classe de Conexão com Banco de Dados
 * Gerencia conexões PDO com MySQL
 */

class Database {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            error_log("Erro de conexão com banco de dados: " . $e->getMessage());
            throw new Exception("Erro de conexão com banco de dados");
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Erro na query: " . $e->getMessage() . " SQL: " . $sql);
            throw new Exception("Erro na execução da query");
        }
    }
    
    public function select($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }
    
    public function selectOne($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }
    
    public function insert($table, $data) {
        $columns = implode(',', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));
        
        $sql = "INSERT INTO {$table} ({$columns}) VALUES ({$placeholders})";
        $this->query($sql, $data);
        
        return $this->connection->lastInsertId();
    }
    
    public function update($table, $data, $where, $whereParams = []) {
        $setClause = [];
        foreach (array_keys($data) as $key) {
            $setClause[] = "{$key} = :{$key}";
        }
        $setClause = implode(', ', $setClause);
        
        $sql = "UPDATE {$table} SET {$setClause} WHERE {$where}";
        $params = array_merge($data, $whereParams);
        
        $stmt = $this->query($sql, $params);
        return $stmt->rowCount();
    }
    
    public function delete($table, $where, $params = []) {
        $sql = "DELETE FROM {$table} WHERE {$where}";
        $stmt = $this->query($sql, $params);
        return $stmt->rowCount();
    }
    
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }
    
    public function commit() {
        return $this->connection->commit();
    }
    
    public function rollback() {
        return $this->connection->rollback();
    }
    
    public function lastInsertId() {
        return $this->connection->lastInsertId();
    }
    
    // Método para executar múltiplas queries (útil para instalação)
    public function executeMultiple($sql) {
        try {
            $this->connection->exec($sql);
            return true;
        } catch (PDOException $e) {
            error_log("Erro na execução múltipla: " . $e->getMessage());
            return false;
        }
    }
    
    // Método para verificar se uma tabela existe
    public function tableExists($tableName) {
        $sql = "SHOW TABLES LIKE :table";
        $result = $this->selectOne($sql, ['table' => $tableName]);
        return !empty($result);
    }
    
    // Método para obter informações da tabela
    public function getTableInfo($tableName) {
        $sql = "DESCRIBE {$tableName}";
        return $this->select($sql);
    }
    
    // Método para backup de dados
    public function backup($tables = []) {
        // Implementação futura para backup
        return true;
    }
    
    // Método para logs de auditoria
    public function log($action, $table, $recordId, $userId = null, $details = null) {
        $logData = [
            'action' => $action,
            'table_name' => $table,
            'record_id' => $recordId,
            'user_id' => $userId,
            'details' => $details ? json_encode($details) : null,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Criar tabela de logs se não existir
        if (!$this->tableExists('audit_logs')) {
            $this->createAuditTable();
        }
        
        return $this->insert('audit_logs', $logData);
    }
    
    private function createAuditTable() {
        $sql = "
            CREATE TABLE audit_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                action VARCHAR(50) NOT NULL,
                table_name VARCHAR(50) NOT NULL,
                record_id INT,
                user_id INT,
                details TEXT,
                ip_address VARCHAR(45),
                user_agent TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_table_record (table_name, record_id),
                INDEX idx_user_action (user_id, action),
                INDEX idx_created_at (created_at)
            )
        ";
        
        $this->connection->exec($sql);
    }
}
?>

