<?php
/**
 * CliniSys - Classe Base para Entidades
 * Fornece funcionalidades comuns para todas as entidades
 */

abstract class BaseEntity {
    protected $db;
    protected $table;
    protected $primaryKey = 'id';
    protected $fillable = [];
    protected $hidden = ['senha'];
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    // Buscar todos os registros
    public function all($conditions = '', $params = []) {
        $sql = "SELECT * FROM {$this->table}";
        if ($conditions) {
            $sql .= " WHERE {$conditions}";
        }
        $sql .= " ORDER BY {$this->primaryKey} DESC";
        
        return $this->db->select($sql, $params);
    }
    
    // Buscar por ID
    public function find($id) {
        $sql = "SELECT * FROM {$this->table} WHERE {$this->primaryKey} = :id";
        return $this->db->selectOne($sql, ['id' => $id]);
    }
    
    // Buscar com condições
    public function where($conditions, $params = []) {
        $sql = "SELECT * FROM {$this->table} WHERE {$conditions}";
        return $this->db->select($sql, $params);
    }
    
    // Buscar um registro com condições
    public function whereOne($conditions, $params = []) {
        $sql = "SELECT * FROM {$this->table} WHERE {$conditions} LIMIT 1";
        return $this->db->selectOne($sql, $params);
    }
    
    // Criar novo registro
    public function create($data) {
        $filteredData = $this->filterFillable($data);
        $id = $this->db->insert($this->table, $filteredData);
        
        // Log da ação
        $this->db->log('CREATE', $this->table, $id, $this->getCurrentUserId(), $filteredData);
        
        return $id;
    }
    
    // Atualizar registro
    public function update($id, $data) {
        $filteredData = $this->filterFillable($data);
        $filteredData['updated_at'] = date('Y-m-d H:i:s');
        
        $result = $this->db->update(
            $this->table, 
            $filteredData, 
            "{$this->primaryKey} = :id", 
            ['id' => $id]
        );
        
        // Log da ação
        if ($result > 0) {
            $this->db->log('UPDATE', $this->table, $id, $this->getCurrentUserId(), $filteredData);
        }
        
        return $result;
    }
    
    // Deletar registro (soft delete se campo 'ativo' existir)
    public function delete($id) {
        // Verificar se a tabela tem campo 'ativo' para soft delete
        $tableInfo = $this->db->getTableInfo($this->table);
        $hasAtivoField = false;
        
        foreach ($tableInfo as $column) {
            if ($column['Field'] === 'ativo') {
                $hasAtivoField = true;
                break;
            }
        }
        
        if ($hasAtivoField) {
            // Soft delete
            $result = $this->update($id, ['ativo' => 0]);
            $action = 'SOFT_DELETE';
        } else {
            // Hard delete
            $result = $this->db->delete($this->table, "{$this->primaryKey} = :id", ['id' => $id]);
            $action = 'DELETE';
        }
        
        // Log da ação
        if ($result > 0) {
            $this->db->log($action, $this->table, $id, $this->getCurrentUserId());
        }
        
        return $result;
    }
    
    // Restaurar registro (para soft delete)
    public function restore($id) {
        $result = $this->update($id, ['ativo' => 1]);
        
        if ($result > 0) {
            $this->db->log('RESTORE', $this->table, $id, $this->getCurrentUserId());
        }
        
        return $result;
    }
    
    // Contar registros
    public function count($conditions = '', $params = []) {
        $sql = "SELECT COUNT(*) as total FROM {$this->table}";
        if ($conditions) {
            $sql .= " WHERE {$conditions}";
        }
        
        $result = $this->db->selectOne($sql, $params);
        return $result['total'] ?? 0;
    }
    
    // Paginação
    public function paginate($page = 1, $perPage = 10, $conditions = '', $params = []) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT * FROM {$this->table}";
        if ($conditions) {
            $sql .= " WHERE {$conditions}";
        }
        $sql .= " ORDER BY {$this->primaryKey} DESC LIMIT {$perPage} OFFSET {$offset}";
        
        $data = $this->db->select($sql, $params);
        $total = $this->count($conditions, $params);
        
        return [
            'data' => $data,
            'total' => $total,
            'page' => $page,
            'per_page' => $perPage,
            'total_pages' => ceil($total / $perPage)
        ];
    }
    
    // Filtrar campos permitidos
    protected function filterFillable($data) {
        if (empty($this->fillable)) {
            return $data;
        }
        
        return array_intersect_key($data, array_flip($this->fillable));
    }
    
    // Remover campos ocultos
    protected function hideFields($data) {
        if (is_array($data) && isset($data[0])) {
            // Array de registros
            return array_map([$this, 'hideFields'], $data);
        }
        
        // Registro único
        foreach ($this->hidden as $field) {
            unset($data[$field]);
        }
        
        return $data;
    }
    
    // Obter ID do usuário atual
    protected function getCurrentUserId() {
        return $_SESSION['user_id'] ?? null;
    }
    
    // Validar dados
    public function validate($data, $rules = []) {
        $errors = [];
        
        foreach ($rules as $field => $rule) {
            $value = $data[$field] ?? null;
            
            if (strpos($rule, 'required') !== false && empty($value)) {
                $errors[$field] = "O campo {$field} é obrigatório";
                continue;
            }
            
            if (strpos($rule, 'email') !== false && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                $errors[$field] = "O campo {$field} deve ser um email válido";
            }
            
            if (preg_match('/min:(\d+)/', $rule, $matches) && strlen($value) < $matches[1]) {
                $errors[$field] = "O campo {$field} deve ter pelo menos {$matches[1]} caracteres";
            }
            
            if (preg_match('/max:(\d+)/', $rule, $matches) && strlen($value) > $matches[1]) {
                $errors[$field] = "O campo {$field} deve ter no máximo {$matches[1]} caracteres";
            }
        }
        
        return $errors;
    }
    
    // Busca com JOIN
    public function join($table, $on, $type = 'INNER') {
        $sql = "SELECT * FROM {$this->table} {$type} JOIN {$table} ON {$on}";
        return $this->db->select($sql);
    }
    
    // Busca com múltiplos JOINs
    public function complexQuery($sql, $params = []) {
        return $this->db->select($sql, $params);
    }
}
?>

