<?php
/**
 * CliniSys - Sistema de Autenticação
 * Gerencia login, logout e controle de acesso
 */

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
        
        // Iniciar sessão se não estiver iniciada
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }
    
    // Fazer login
    public function login($email, $senha) {
        try {
            $sql = "SELECT u.*, m.crm, m.especialidade_id, e.nome as especialidade_nome 
                    FROM usuarios u 
                    LEFT JOIN medicos m ON u.id = m.usuario_id 
                    LEFT JOIN especialidades e ON m.especialidade_id = e.id 
                    WHERE u.email = :email AND u.ativo = 1";
            
            $user = $this->db->selectOne($sql, ['email' => $email]);
            
            if ($user && password_verify($senha, $user['senha'])) {
                // Criar sessão
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_name'] = $user['nome'];
                $_SESSION['user_email'] = $user['email'];
                $_SESSION['user_type'] = $user['tipo'];
                $_SESSION['login_time'] = time();
                
                // Se for médico, adicionar informações específicas
                if ($user['tipo'] === 'medico' && $user['crm']) {
                    $_SESSION['medico_id'] = $user['id'];
                    $_SESSION['crm'] = $user['crm'];
                    $_SESSION['especialidade_id'] = $user['especialidade_id'];
                    $_SESSION['especialidade_nome'] = $user['especialidade_nome'];
                }
                
                // Log do login
                $this->db->log('LOGIN', 'usuarios', $user['id'], $user['id']);
                
                return [
                    'success' => true,
                    'user' => $this->getUserData($user),
                    'message' => 'Login realizado com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Email ou senha incorretos'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro interno do servidor'
            ];
        }
    }
    
    // Fazer logout
    public function logout() {
        if ($this->isLoggedIn()) {
            // Log do logout
            $this->db->log('LOGOUT', 'usuarios', $_SESSION['user_id'], $_SESSION['user_id']);
        }
        
        // Destruir sessão
        session_destroy();
        
        return [
            'success' => true,
            'message' => 'Logout realizado com sucesso'
        ];
    }
    
    // Verificar se está logado
    public function isLoggedIn() {
        return isset($_SESSION['user_id']) && $this->isSessionValid();
    }
    
    // Verificar se a sessão é válida
    private function isSessionValid() {
        if (!isset($_SESSION['login_time'])) {
            return false;
        }
        
        // Verificar timeout da sessão
        if (time() - $_SESSION['login_time'] > SESSION_TIMEOUT) {
            $this->logout();
            return false;
        }
        
        return true;
    }
    
    // Obter usuário atual
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        $sql = "SELECT u.*, m.crm, m.especialidade_id, e.nome as especialidade_nome 
                FROM usuarios u 
                LEFT JOIN medicos m ON u.id = m.usuario_id 
                LEFT JOIN especialidades e ON m.especialidade_id = e.id 
                WHERE u.id = :id";
        
        $user = $this->db->selectOne($sql, ['id' => $_SESSION['user_id']]);
        
        return $user ? $this->getUserData($user) : null;
    }
    
    // Verificar permissão
    public function hasPermission($permission) {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        $userType = $_SESSION['user_type'];
        
        $permissions = [
            'admin' => ['*'], // Admin tem todas as permissões
            'medico' => [
                'view_agendamentos',
                'update_agendamentos',
                'view_pacientes',
                'update_pacientes',
                'view_fila',
                'call_patient'
            ],
            'recepcionista' => [
                'view_agendamentos',
                'create_agendamentos',
                'update_agendamentos',
                'view_pacientes',
                'create_pacientes',
                'update_pacientes',
                'manage_fila',
                'call_patient'
            ]
        ];
        
        $userPermissions = $permissions[$userType] ?? [];
        
        return in_array('*', $userPermissions) || in_array($permission, $userPermissions);
    }
    
    // Middleware de autenticação
    public function requireAuth() {
        if (!$this->isLoggedIn()) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Acesso não autorizado'
            ]);
            exit;
        }
    }
    
    // Middleware de permissão
    public function requirePermission($permission) {
        $this->requireAuth();
        
        if (!$this->hasPermission($permission)) {
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'message' => 'Permissão insuficiente'
            ]);
            exit;
        }
    }
    
    // Criar novo usuário
    public function createUser($data) {
        try {
            // Validar dados
            $errors = $this->validateUserData($data);
            if (!empty($errors)) {
                return [
                    'success' => false,
                    'errors' => $errors
                ];
            }
            
            // Verificar se email já existe
            $existingUser = $this->db->selectOne(
                "SELECT id FROM usuarios WHERE email = :email",
                ['email' => $data['email']]
            );
            
            if ($existingUser) {
                return [
                    'success' => false,
                    'message' => 'Email já está em uso'
                ];
            }
            
            // Hash da senha
            $data['senha'] = password_hash($data['senha'], PASSWORD_DEFAULT);
            
            // Criar usuário
            $userId = $this->db->insert('usuarios', [
                'nome' => $data['nome'],
                'email' => $data['email'],
                'senha' => $data['senha'],
                'tipo' => $data['tipo']
            ]);
            
            // Se for médico, criar registro na tabela médicos
            if ($data['tipo'] === 'medico' && isset($data['crm'])) {
                $this->db->insert('medicos', [
                    'usuario_id' => $userId,
                    'crm' => $data['crm'],
                    'especialidade_id' => $data['especialidade_id'] ?? null,
                    'telefone' => $data['telefone'] ?? null
                ]);
            }
            
            return [
                'success' => true,
                'user_id' => $userId,
                'message' => 'Usuário criado com sucesso'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao criar usuário'
            ];
        }
    }
    
    // Alterar senha
    public function changePassword($userId, $senhaAtual, $novaSenha) {
        try {
            // Verificar senha atual
            $user = $this->db->selectOne(
                "SELECT senha FROM usuarios WHERE id = :id",
                ['id' => $userId]
            );
            
            if (!$user || !password_verify($senhaAtual, $user['senha'])) {
                return [
                    'success' => false,
                    'message' => 'Senha atual incorreta'
                ];
            }
            
            // Atualizar senha
            $novaSenhaHash = password_hash($novaSenha, PASSWORD_DEFAULT);
            $this->db->update(
                'usuarios',
                ['senha' => $novaSenhaHash],
                'id = :id',
                ['id' => $userId]
            );
            
            // Log da ação
            $this->db->log('CHANGE_PASSWORD', 'usuarios', $userId, $userId);
            
            return [
                'success' => true,
                'message' => 'Senha alterada com sucesso'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao alterar senha'
            ];
        }
    }
    
    // Validar dados do usuário
    private function validateUserData($data) {
        $errors = [];
        
        if (empty($data['nome'])) {
            $errors['nome'] = 'Nome é obrigatório';
        }
        
        if (empty($data['email']) || !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Email válido é obrigatório';
        }
        
        if (empty($data['senha']) || strlen($data['senha']) < 6) {
            $errors['senha'] = 'Senha deve ter pelo menos 6 caracteres';
        }
        
        if (empty($data['tipo']) || !in_array($data['tipo'], ['admin', 'medico', 'recepcionista'])) {
            $errors['tipo'] = 'Tipo de usuário inválido';
        }
        
        if ($data['tipo'] === 'medico' && empty($data['crm'])) {
            $errors['crm'] = 'CRM é obrigatório para médicos';
        }
        
        return $errors;
    }
    
    // Formatar dados do usuário para retorno
    private function getUserData($user) {
        unset($user['senha']); // Remover senha dos dados retornados
        return $user;
    }
    
    // Reset de senha (para implementação futura)
    public function requestPasswordReset($email) {
        // Implementar envio de email com token de reset
        return [
            'success' => true,
            'message' => 'Instruções enviadas por email'
        ];
    }
}
?>

