<?php
/**
 * CliniSys - Classe Agendamento
 * Gerencia operações relacionadas aos agendamentos
 */

require_once 'BaseEntity.php';

class Agendamento extends BaseEntity {
    protected $table = 'agendamentos';
    protected $fillable = [
        'paciente_id', 'medico_id', 'data_agendamento', 'duracao', 
        'tipo_consulta', 'status', 'observacoes', 'valor'
    ];
    
    // Criar novo agendamento
    public function createAgendamento($data) {
        // Validar dados
        $errors = $this->validateAgendamento($data);
        if (!empty($errors)) {
            return ['success' => false, 'errors' => $errors];
        }
        
        // Verificar disponibilidade do médico
        $medico = new Medico();
        if (!$medico->isDisponivel($data['medico_id'], $data['data_agendamento'], $data['duracao'] ?? 30)) {
            return ['success' => false, 'message' => 'Horário não disponível'];
        }
        
        try {
            $agendamentoId = $this->create($data);
            
            return [
                'success' => true,
                'agendamento_id' => $agendamentoId,
                'message' => 'Agendamento criado com sucesso'
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Erro ao criar agendamento'];
        }
    }
    
    // Obter agendamentos com detalhes completos
    public function getAllWithDetails($filters = []) {
        $where = ['1 = 1'];
        $params = [];
        
        // Aplicar filtros
        if (!empty($filters['data_inicio'])) {
            $where[] = 'a.data_agendamento >= :data_inicio';
            $params['data_inicio'] = $filters['data_inicio'];
        }
        
        if (!empty($filters['data_fim'])) {
            $where[] = 'a.data_agendamento <= :data_fim';
            $params['data_fim'] = $filters['data_fim'];
        }
        
        if (!empty($filters['medico_id'])) {
            $where[] = 'a.medico_id = :medico_id';
            $params['medico_id'] = $filters['medico_id'];
        }
        
        if (!empty($filters['paciente_id'])) {
            $where[] = 'a.paciente_id = :paciente_id';
            $params['paciente_id'] = $filters['paciente_id'];
        }
        
        if (!empty($filters['status'])) {
            $where[] = 'a.status = :status';
            $params['status'] = $filters['status'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $sql = "
            SELECT a.*, 
                   p.nome as paciente_nome, p.telefone as paciente_telefone, 
                   p.celular as paciente_celular, p.convenio as paciente_convenio,
                   u.nome as medico_nome, m.crm as medico_crm,
                   e.nome as especialidade_nome, e.cor as especialidade_cor
            FROM agendamentos a
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE {$whereClause}
            ORDER BY a.data_agendamento DESC
        ";
        
        return $this->db->select($sql, $params);
    }
    
    // Obter agendamentos para o calendário (formato FullCalendar)
    public function getForCalendar($start, $end, $medicoId = null) {
        $where = ['a.data_agendamento BETWEEN :start AND :end'];
        $params = ['start' => $start, 'end' => $end];
        
        if ($medicoId) {
            $where[] = 'a.medico_id = :medico_id';
            $params['medico_id'] = $medicoId;
        }
        
        $whereClause = implode(' AND ', $where);
        
        $sql = "
            SELECT a.id, a.data_agendamento as start, a.duracao, a.status, a.tipo_consulta,
                   p.nome as paciente_nome,
                   u.nome as medico_nome,
                   e.nome as especialidade_nome, e.cor as color
            FROM agendamentos a
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE {$whereClause}
            AND a.status NOT IN ('cancelado')
        ";
        
        $agendamentos = $this->db->select($sql, $params);
        
        // Formatar para FullCalendar
        $events = [];
        foreach ($agendamentos as $agendamento) {
            $start = new DateTime($agendamento['start']);
            $end = clone $start;
            $end->add(new DateInterval('PT' . $agendamento['duracao'] . 'M'));
            
            $events[] = [
                'id' => $agendamento['id'],
                'title' => $agendamento['paciente_nome'] . ' - ' . $agendamento['medico_nome'],
                'start' => $start->format('c'),
                'end' => $end->format('c'),
                'color' => $agendamento['color'] ?? '#007bff',
                'extendedProps' => [
                    'paciente' => $agendamento['paciente_nome'],
                    'medico' => $agendamento['medico_nome'],
                    'especialidade' => $agendamento['especialidade_nome'],
                    'tipo' => $agendamento['tipo_consulta'],
                    'status' => $agendamento['status']
                ]
            ];
        }
        
        return $events;
    }
    
    // Confirmar agendamento
    public function confirmar($agendamentoId) {
        return $this->update($agendamentoId, ['status' => 'confirmado']);
    }
    
    // Cancelar agendamento
    public function cancelar($agendamentoId, $motivo = null) {
        $data = ['status' => 'cancelado'];
        if ($motivo) {
            $data['observacoes'] = $motivo;
        }
        
        return $this->update($agendamentoId, $data);
    }
    
    // Marcar como em atendimento
    public function iniciarAtendimento($agendamentoId) {
        return $this->update($agendamentoId, ['status' => 'em_atendimento']);
    }
    
    // Finalizar atendimento
    public function finalizarAtendimento($agendamentoId, $observacoes = null) {
        $data = ['status' => 'finalizado'];
        if ($observacoes) {
            $data['observacoes'] = $observacoes;
        }
        
        return $this->update($agendamentoId, $data);
    }
    
    // Marcar falta
    public function marcarFalta($agendamentoId) {
        return $this->update($agendamentoId, ['status' => 'faltou']);
    }
    
    // Reagendar consulta
    public function reagendar($agendamentoId, $novaDataHora) {
        $agendamento = $this->find($agendamentoId);
        if (!$agendamento) {
            return ['success' => false, 'message' => 'Agendamento não encontrado'];
        }
        
        // Verificar disponibilidade do médico no novo horário
        $medico = new Medico();
        if (!$medico->isDisponivel($agendamento['medico_id'], $novaDataHora, $agendamento['duracao'])) {
            return ['success' => false, 'message' => 'Novo horário não disponível'];
        }
        
        $result = $this->update($agendamentoId, [
            'data_agendamento' => $novaDataHora,
            'status' => 'agendado'
        ]);
        
        if ($result) {
            return ['success' => true, 'message' => 'Agendamento reagendado com sucesso'];
        } else {
            return ['success' => false, 'message' => 'Erro ao reagendar'];
        }
    }
    
    // Obter agendamentos do dia
    public function getAgendamentosDoDia($data = null, $medicoId = null) {
        if (!$data) {
            $data = date('Y-m-d');
        }
        
        $where = ['DATE(a.data_agendamento) = :data'];
        $params = ['data' => $data];
        
        if ($medicoId) {
            $where[] = 'a.medico_id = :medico_id';
            $params['medico_id'] = $medicoId;
        }
        
        $whereClause = implode(' AND ', $where);
        
        $sql = "
            SELECT a.*, 
                   p.nome as paciente_nome, p.telefone as paciente_telefone,
                   u.nome as medico_nome,
                   e.nome as especialidade_nome
            FROM agendamentos a
            INNER JOIN pacientes p ON a.paciente_id = p.id
            INNER JOIN medicos m ON a.medico_id = m.id
            INNER JOIN usuarios u ON m.usuario_id = u.id
            LEFT JOIN especialidades e ON m.especialidade_id = e.id
            WHERE {$whereClause}
            AND a.status NOT IN ('cancelado')
            ORDER BY a.data_agendamento
        ";
        
        return $this->db->select($sql, $params);
    }
    
    // Obter estatísticas de agendamentos
    public function getEstatisticas($periodo = '30 days', $medicoId = null) {
        $where = ['a.data_agendamento >= DATE_SUB(NOW(), INTERVAL ' . $periodo . ')'];
        $params = [];
        
        if ($medicoId) {
            $where[] = 'a.medico_id = :medico_id';
            $params['medico_id'] = $medicoId;
        }
        
        $whereClause = implode(' AND ', $where);
        
        $sql = "
            SELECT 
                COUNT(*) as total,
                COUNT(CASE WHEN status = 'agendado' THEN 1 END) as agendados,
                COUNT(CASE WHEN status = 'confirmado' THEN 1 END) as confirmados,
                COUNT(CASE WHEN status = 'finalizado' THEN 1 END) as finalizados,
                COUNT(CASE WHEN status = 'cancelado' THEN 1 END) as cancelados,
                COUNT(CASE WHEN status = 'faltou' THEN 1 END) as faltas,
                AVG(valor) as valor_medio,
                SUM(CASE WHEN status = 'finalizado' THEN valor ELSE 0 END) as faturamento
            FROM agendamentos a
            WHERE {$whereClause}
        ";
        
        return $this->db->selectOne($sql, $params);
    }
    
    // Validar dados do agendamento
    public function validateAgendamento($data) {
        $errors = [];
        
        if (empty($data['paciente_id'])) {
            $errors['paciente_id'] = 'Paciente é obrigatório';
        }
        
        if (empty($data['medico_id'])) {
            $errors['medico_id'] = 'Médico é obrigatório';
        }
        
        if (empty($data['data_agendamento'])) {
            $errors['data_agendamento'] = 'Data e hora são obrigatórias';
        } else {
            $dataAgendamento = new DateTime($data['data_agendamento']);
            $agora = new DateTime();
            
            // Verificar se a data não é no passado
            if ($dataAgendamento < $agora) {
                $errors['data_agendamento'] = 'Data não pode ser no passado';
            }
            
            // Verificar antecedência mínima
            $antecedenciaMinima = clone $agora;
            $antecedenciaMinima->add(new DateInterval('PT' . ANTECEDENCIA_MINIMA . 'H'));
            
            if ($dataAgendamento < $antecedenciaMinima) {
                $errors['data_agendamento'] = 'Agendamento deve ter antecedência mínima de ' . ANTECEDENCIA_MINIMA . ' hora(s)';
            }
        }
        
        if (!empty($data['duracao']) && ($data['duracao'] < 15 || $data['duracao'] > 120)) {
            $errors['duracao'] = 'Duração deve estar entre 15 e 120 minutos';
        }
        
        if (!empty($data['valor']) && $data['valor'] < 0) {
            $errors['valor'] = 'Valor não pode ser negativo';
        }
        
        return $errors;
    }
    
    // Buscar conflitos de horário
    public function findConflitos($medicoId, $dataHora, $duracao, $excludeId = null) {
        $dataInicio = new DateTime($dataHora);
        $dataFim = clone $dataInicio;
        $dataFim->add(new DateInterval('PT' . $duracao . 'M'));
        
        $where = [
            'medico_id = :medico_id',
            'status NOT IN ("cancelado", "faltou")',
            '(',
            '(data_agendamento <= :data_inicio AND DATE_ADD(data_agendamento, INTERVAL duracao MINUTE) > :data_inicio)',
            'OR',
            '(data_agendamento < :data_fim AND data_agendamento >= :data_inicio)',
            ')'
        ];
        
        $params = [
            'medico_id' => $medicoId,
            'data_inicio' => $dataInicio->format('Y-m-d H:i:s'),
            'data_fim' => $dataFim->format('Y-m-d H:i:s')
        ];
        
        if ($excludeId) {
            $where[] = 'id != :exclude_id';
            $params['exclude_id'] = $excludeId;
        }
        
        $whereClause = implode(' ', $where);
        
        return $this->where($whereClause, $params);
    }
}
?>

