<?php
/**
 * CliniSys - API de Pacientes
 * Endpoints para gerenciar pacientes
 */

require_once '../includes/autoload.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGet($path, $auth);
            break;
        case 'POST':
            handlePost($path, $auth);
            break;
        case 'PUT':
            handlePut($path, $auth);
            break;
        case 'DELETE':
            handleDelete($path, $auth);
            break;
        default:
            jsonResponse(['success' => false, 'message' => 'Método não permitido'], 405);
    }
} catch (Exception $e) {
    logError('Erro na API de pacientes: ' . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
}

function handleGet($action, $auth) {
    $paciente = new Paciente();
    
    switch ($action) {
        case 'list':
            $auth->requirePermission('view_pacientes');
            
            $page = $_GET['page'] ?? 1;
            $perPage = $_GET['per_page'] ?? 20;
            $search = $_GET['search'] ?? '';
            
            if ($search) {
                $pacientes = $paciente->searchByName($search);
                $result = [
                    'data' => $pacientes,
                    'total' => count($pacientes),
                    'page' => 1,
                    'per_page' => count($pacientes),
                    'total_pages' => 1
                ];
            } else {
                $result = $paciente->paginate($page, $perPage, 'ativo = 1');
            }
            
            jsonResponse(['success' => true, 'data' => $result]);
            break;
            
        case 'show':
            $auth->requirePermission('view_pacientes');
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $dados = $paciente->find($id);
            if (!$dados) {
                jsonResponse(['success' => false, 'message' => 'Paciente não encontrado'], 404);
            }
            
            jsonResponse(['success' => true, 'data' => $dados]);
            break;
            
        case 'search':
            $auth->requirePermission('view_pacientes');
            
            $termo = $_GET['q'] ?? '';
            if (strlen($termo) < 2) {
                jsonResponse(['success' => false, 'message' => 'Termo de busca deve ter pelo menos 2 caracteres'], 400);
            }
            
            $resultados = $paciente->searchByName($termo);
            jsonResponse(['success' => true, 'data' => $resultados]);
            break;
            
        case 'by-cpf':
            $auth->requirePermission('view_pacientes');
            
            $cpf = $_GET['cpf'] ?? '';
            if (empty($cpf)) {
                jsonResponse(['success' => false, 'message' => 'CPF obrigatório'], 400);
            }
            
            $dados = $paciente->findByCpf($cpf);
            if (!$dados) {
                jsonResponse(['success' => false, 'message' => 'Paciente não encontrado'], 404);
            }
            
            jsonResponse(['success' => true, 'data' => $dados]);
            break;
            
        case 'appointments':
            $auth->requirePermission('view_pacientes');
            
            $id = $_GET['id'] ?? null;
            $limit = $_GET['limit'] ?? 10;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $agendamentos = $paciente->getAgendamentos($id, $limit);
            jsonResponse(['success' => true, 'data' => $agendamentos]);
            break;
            
        case 'next-appointments':
            $auth->requirePermission('view_pacientes');
            
            $id = $_GET['id'] ?? null;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $agendamentos = $paciente->getProximosAgendamentos($id);
            jsonResponse(['success' => true, 'data' => $agendamentos]);
            break;
            
        case 'stats':
            $auth->requirePermission('view_pacientes');
            
            $id = $_GET['id'] ?? null;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $stats = $paciente->getEstatisticas($id);
            jsonResponse(['success' => true, 'data' => $stats]);
            break;
            
        case 'birthdays':
            $auth->requirePermission('view_pacientes');
            
            $aniversariantes = $paciente->getAniversariantes();
            jsonResponse(['success' => true, 'data' => $aniversariantes]);
            break;
            
        case 'by-age':
            $auth->requirePermission('view_pacientes');
            
            $idadeMin = $_GET['idade_min'] ?? 0;
            $idadeMax = $_GET['idade_max'] ?? 120;
            
            $pacientes = $paciente->findByIdade($idadeMin, $idadeMax);
            jsonResponse(['success' => true, 'data' => $pacientes]);
            break;
            
        case 'by-insurance':
            $auth->requirePermission('view_pacientes');
            
            $convenio = $_GET['convenio'] ?? '';
            if (empty($convenio)) {
                jsonResponse(['success' => false, 'message' => 'Convênio obrigatório'], 400);
            }
            
            $pacientes = $paciente->findByConvenio($convenio);
            jsonResponse(['success' => true, 'data' => $pacientes]);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePost($action, $auth) {
    $paciente = new Paciente();
    $data = getJsonInput();
    
    switch ($action) {
        case 'create':
            $auth->requirePermission('create_pacientes');
            
            $errors = $paciente->validatePaciente($data);
            if (!empty($errors)) {
                jsonResponse(['success' => false, 'errors' => $errors], 400);
            }
            
            // Formatar CPF se fornecido
            if (!empty($data['cpf'])) {
                $data['cpf'] = $paciente->formatarCpf($data['cpf']);
            }
            
            // Formatar telefones se fornecidos
            if (!empty($data['telefone'])) {
                $data['telefone'] = $paciente->formatarTelefone($data['telefone']);
            }
            if (!empty($data['celular'])) {
                $data['celular'] = $paciente->formatarTelefone($data['celular']);
            }
            
            $id = $paciente->create($data);
            
            if ($id) {
                jsonResponse(['success' => true, 'id' => $id, 'message' => 'Paciente criado com sucesso'], 201);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao criar paciente'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePut($action, $auth) {
    $paciente = new Paciente();
    $data = getJsonInput();
    
    switch ($action) {
        case 'update':
            $auth->requirePermission('update_pacientes');
            
            $id = $data['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            unset($data['id']);
            
            $errors = $paciente->validatePaciente($data, true);
            if (!empty($errors)) {
                jsonResponse(['success' => false, 'errors' => $errors], 400);
            }
            
            // Formatar telefones se fornecidos
            if (!empty($data['telefone'])) {
                $data['telefone'] = $paciente->formatarTelefone($data['telefone']);
            }
            if (!empty($data['celular'])) {
                $data['celular'] = $paciente->formatarTelefone($data['celular']);
            }
            
            $result = $paciente->update($id, $data);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Paciente atualizado com sucesso']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao atualizar paciente'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handleDelete($action, $auth) {
    $paciente = new Paciente();
    
    switch ($action) {
        case 'delete':
            $auth->requirePermission('*'); // Apenas admin
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $paciente->delete($id);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Paciente removido com sucesso']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao remover paciente'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}
?>

