<?php
/**
 * CliniSys - API de Médicos
 * Endpoints para gerenciar médicos
 */

require_once '../includes/autoload.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGet($path, $auth);
            break;
        case 'POST':
            handlePost($path, $auth);
            break;
        case 'PUT':
            handlePut($path, $auth);
            break;
        case 'DELETE':
            handleDelete($path, $auth);
            break;
        default:
            jsonResponse(['success' => false, 'message' => 'Método não permitido'], 405);
    }
} catch (Exception $e) {
    logError('Erro na API de médicos: ' . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
}

function handleGet($action, $auth) {
    $medico = new Medico();
    
    switch ($action) {
        case 'list':
            $auth->requirePermission('view_agendamentos');
            
            $medicos = $medico->getAllWithDetails();
            jsonResponse(['success' => true, 'data' => $medicos]);
            break;
            
        case 'show':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $dados = $medico->find($id);
            if (!$dados) {
                jsonResponse(['success' => false, 'message' => 'Médico não encontrado'], 404);
            }
            
            jsonResponse(['success' => true, 'data' => $dados]);
            break;
            
        case 'by-specialty':
            $auth->requirePermission('view_agendamentos');
            
            $especialidadeId = $_GET['especialidade_id'] ?? null;
            if (!$especialidadeId) {
                jsonResponse(['success' => false, 'message' => 'ID da especialidade obrigatório'], 400);
            }
            
            $medicos = $medico->findByEspecialidade($especialidadeId);
            jsonResponse(['success' => true, 'data' => $medicos]);
            break;
            
        case 'schedule':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $horarios = $medico->getHorariosFuncionamento($id);
            jsonResponse(['success' => true, 'data' => $horarios]);
            break;
            
        case 'agenda':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            $dataInicio = $_GET['data_inicio'] ?? date('Y-m-d');
            $dataFim = $_GET['data_fim'] ?? date('Y-m-d', strtotime('+7 days'));
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $agenda = $medico->getAgenda($id, $dataInicio, $dataFim);
            jsonResponse(['success' => true, 'data' => $agenda]);
            break;
            
        case 'available-times':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            $data = $_GET['data'] ?? null;
            $duracao = $_GET['duracao'] ?? 30;
            
            if (!$id || !$data) {
                jsonResponse(['success' => false, 'message' => 'ID e data são obrigatórios'], 400);
            }
            
            $horarios = $medico->getHorariosDisponiveis($id, $data, $duracao);
            jsonResponse(['success' => true, 'data' => $horarios]);
            break;
            
        case 'stats':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            $periodo = $_GET['periodo'] ?? '30 days';
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $stats = $medico->getEstatisticas($id, $periodo);
            jsonResponse(['success' => true, 'data' => $stats]);
            break;
            
        case 'next-appointments':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            $limit = $_GET['limit'] ?? 5;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $agendamentos = $medico->getProximosAgendamentos($id, $limit);
            jsonResponse(['success' => true, 'data' => $agendamentos]);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePost($action, $auth) {
    $medico = new Medico();
    $data = getJsonInput();
    
    switch ($action) {
        case 'create':
            $auth->requirePermission('*'); // Apenas admin
            
            $errors = $medico->validateMedico($data);
            if (!empty($errors)) {
                jsonResponse(['success' => false, 'errors' => $errors], 400);
            }
            
            $id = $medico->create($data);
            
            if ($id) {
                jsonResponse(['success' => true, 'id' => $id, 'message' => 'Médico criado com sucesso'], 201);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao criar médico'], 400);
            }
            break;
            
        case 'set-schedule':
            $auth->requirePermission('*'); // Apenas admin ou próprio médico
            
            $id = $data['medico_id'] ?? null;
            $horarios = $data['horarios'] ?? [];
            
            if (!$id || empty($horarios)) {
                jsonResponse(['success' => false, 'message' => 'ID do médico e horários são obrigatórios'], 400);
            }
            
            $result = $medico->setHorariosFuncionamento($id, $horarios);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Horários definidos com sucesso']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao definir horários'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePut($action, $auth) {
    $medico = new Medico();
    $data = getJsonInput();
    
    switch ($action) {
        case 'update':
            $auth->requirePermission('*'); // Apenas admin
            
            $id = $data['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            unset($data['id']);
            
            $errors = $medico->validateMedico($data, true);
            if (!empty($errors)) {
                jsonResponse(['success' => false, 'errors' => $errors], 400);
            }
            
            $result = $medico->update($id, $data);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Médico atualizado com sucesso']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao atualizar médico'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handleDelete($action, $auth) {
    $medico = new Medico();
    
    switch ($action) {
        case 'delete':
            $auth->requirePermission('*'); // Apenas admin
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $medico->delete($id);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Médico removido com sucesso']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao remover médico'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}
?>

