<?php
/**
 * CliniSys - API de Fila de Espera
 * Endpoints para gerenciar a fila de espera
 */

require_once '../includes/autoload.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGet($path, $auth);
            break;
        case 'POST':
            handlePost($path, $auth);
            break;
        case 'PUT':
            handlePut($path, $auth);
            break;
        case 'DELETE':
            handleDelete($path, $auth);
            break;
        default:
            jsonResponse(['success' => false, 'message' => 'Método não permitido'], 405);
    }
} catch (Exception $e) {
    logError('Erro na API de fila de espera: ' . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
}

function handleGet($action, $auth) {
    $fila = new FilaEspera();
    
    switch ($action) {
        case 'current':
            $auth->requirePermission('view_fila');
            
            $filaAtual = $fila->getFilaAtual();
            jsonResponse(['success' => true, 'data' => $filaAtual]);
            break;
            
        case 'stats':
            $auth->requirePermission('view_fila');
            
            $data = $_GET['data'] ?? date('Y-m-d');
            $stats = $fila->getEstatisticas($data);
            jsonResponse(['success' => true, 'data' => $stats]);
            break;
            
        case 'tv-calls':
            // Não requer autenticação para TV
            $chamadas = $fila->getChamadasTV();
            jsonResponse(['success' => true, 'data' => $chamadas]);
            break;
            
        case 'next-password':
            $auth->requirePermission('view_fila');
            
            $hoje = date('Y-m-d');
            $sql = "SELECT MAX(numero_senha) as ultimo_numero 
                    FROM fila_espera 
                    WHERE DATE(data_chegada) = :hoje";
            
            $db = Database::getInstance();
            $resultado = $db->selectOne($sql, ['hoje' => $hoje]);
            $proximoNumero = ($resultado['ultimo_numero'] ?? 0) + 1;
            
            jsonResponse(['success' => true, 'data' => ['proximo_numero' => $proximoNumero]]);
            break;
            
        case 'by-appointment':
            $auth->requirePermission('view_fila');
            
            $agendamentoId = $_GET['agendamento_id'] ?? null;
            if (!$agendamentoId) {
                jsonResponse(['success' => false, 'message' => 'ID do agendamento obrigatório'], 400);
            }
            
            $filaItem = $fila->whereOne('agendamento_id = :agendamento_id AND status != "finalizado"', 
                                     ['agendamento_id' => $agendamentoId]);
            
            if ($filaItem) {
                jsonResponse(['success' => true, 'data' => $filaItem]);
            } else {
                jsonResponse(['success' => false, 'message' => 'Paciente não encontrado na fila'], 404);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePost($action, $auth) {
    $fila = new FilaEspera();
    $data = getJsonInput();
    
    switch ($action) {
        case 'add':
            $auth->requirePermission('manage_fila');
            
            $agendamentoId = $data['agendamento_id'] ?? null;
            $prioridade = $data['prioridade'] ?? 'normal';
            
            if (!$agendamentoId) {
                jsonResponse(['success' => false, 'message' => 'ID do agendamento obrigatório'], 400);
            }
            
            $result = $fila->adicionarPaciente($agendamentoId, $prioridade);
            
            if ($result['success']) {
                jsonResponse($result, 201);
            } else {
                jsonResponse($result, 400);
            }
            break;
            
        case 'call-next':
            $auth->requirePermission('call_patient');
            
            $consultorio = $data['consultorio'] ?? 'Consultório 1';
            
            $result = $fila->chamarProximo($consultorio);
            
            if ($result['success']) {
                jsonResponse($result);
            } else {
                jsonResponse($result, 400);
            }
            break;
            
        case 'confirm-attendance':
            $auth->requirePermission('manage_fila');
            
            $filaId = $data['fila_id'] ?? null;
            
            if (!$filaId) {
                jsonResponse(['success' => false, 'message' => 'ID da fila obrigatório'], 400);
            }
            
            $result = $fila->confirmarAtendimento($filaId);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Atendimento confirmado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao confirmar atendimento'], 400);
            }
            break;
            
        case 'finish-attendance':
            $auth->requirePermission('manage_fila');
            
            $filaId = $data['fila_id'] ?? null;
            
            if (!$filaId) {
                jsonResponse(['success' => false, 'message' => 'ID da fila obrigatório'], 400);
            }
            
            $result = $fila->finalizarAtendimento($filaId);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Atendimento finalizado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao finalizar atendimento'], 400);
            }
            break;
            
        case 'recall':
            $auth->requirePermission('call_patient');
            
            $filaId = $data['fila_id'] ?? null;
            $consultorio = $data['consultorio'] ?? 'Consultório 1';
            
            if (!$filaId) {
                jsonResponse(['success' => false, 'message' => 'ID da fila obrigatório'], 400);
            }
            
            $result = $fila->remarcarChamada($filaId, $consultorio);
            jsonResponse($result);
            break;
            
        case 'mark-calls-shown':
            // Não requer autenticação para TV
            $ids = $data['ids'] ?? [];
            
            if (empty($ids)) {
                jsonResponse(['success' => false, 'message' => 'IDs obrigatórios'], 400);
            }
            
            $result = $fila->marcarChamadasExibidas($ids);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Chamadas marcadas como exibidas']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao marcar chamadas'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePut($action, $auth) {
    $fila = new FilaEspera();
    $data = getJsonInput();
    
    switch ($action) {
        case 'update-priority':
            $auth->requirePermission('manage_fila');
            
            $filaId = $data['fila_id'] ?? null;
            $prioridade = $data['prioridade'] ?? null;
            
            if (!$filaId || !$prioridade) {
                jsonResponse(['success' => false, 'message' => 'ID da fila e prioridade obrigatórios'], 400);
            }
            
            if (!in_array($prioridade, ['normal', 'preferencial', 'urgente'])) {
                jsonResponse(['success' => false, 'message' => 'Prioridade inválida'], 400);
            }
            
            $result = $fila->update($filaId, ['prioridade' => $prioridade]);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Prioridade atualizada']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao atualizar prioridade'], 400);
            }
            break;
            
        case 'update-status':
            $auth->requirePermission('manage_fila');
            
            $filaId = $data['fila_id'] ?? null;
            $status = $data['status'] ?? null;
            
            if (!$filaId || !$status) {
                jsonResponse(['success' => false, 'message' => 'ID da fila e status obrigatórios'], 400);
            }
            
            if (!in_array($status, ['aguardando', 'chamado', 'atendendo', 'finalizado'])) {
                jsonResponse(['success' => false, 'message' => 'Status inválido'], 400);
            }
            
            $updateData = ['status' => $status];
            
            // Adicionar timestamp baseado no status
            switch ($status) {
                case 'chamado':
                    $updateData['chamado_em'] = date('Y-m-d H:i:s');
                    break;
                case 'atendendo':
                    $updateData['atendido_em'] = date('Y-m-d H:i:s');
                    break;
                case 'finalizado':
                    $updateData['finalizado_em'] = date('Y-m-d H:i:s');
                    break;
            }
            
            $result = $fila->update($filaId, $updateData);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Status atualizado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao atualizar status'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handleDelete($action, $auth) {
    $fila = new FilaEspera();
    
    switch ($action) {
        case 'remove':
            $auth->requirePermission('manage_fila');
            
            $filaId = $_GET['fila_id'] ?? null;
            $motivo = $_GET['motivo'] ?? 'Removido da fila';
            
            if (!$filaId) {
                jsonResponse(['success' => false, 'message' => 'ID da fila obrigatório'], 400);
            }
            
            $result = $fila->removerDaFila($filaId, $motivo);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Paciente removido da fila']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao remover paciente'], 400);
            }
            break;
            
        case 'clean-old-calls':
            $auth->requirePermission('*'); // Apenas admin
            
            $dias = $_GET['dias'] ?? 7;
            
            $result = $fila->limparChamadasAntigas($dias);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Chamadas antigas removidas']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao limpar chamadas'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}
?>

