<?php
/**
 * CliniSys - API de Autenticação
 * Endpoints para login, logout e gerenciamento de usuários
 */

require_once '../includes/autoload.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGet($path, $auth);
            break;
        case 'POST':
            handlePost($path, $auth);
            break;
        case 'PUT':
            handlePut($path, $auth);
            break;
        default:
            jsonResponse(['success' => false, 'message' => 'Método não permitido'], 405);
    }
} catch (Exception $e) {
    logError('Erro na API de autenticação: ' . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
}

function handleGet($action, $auth) {
    switch ($action) {
        case 'me':
            $auth->requireAuth();
            
            $user = $auth->getCurrentUser();
            if ($user) {
                jsonResponse(['success' => true, 'user' => $user]);
            } else {
                jsonResponse(['success' => false, 'message' => 'Usuário não encontrado'], 404);
            }
            break;
            
        case 'check':
            $isLoggedIn = $auth->isLoggedIn();
            jsonResponse(['success' => true, 'logged_in' => $isLoggedIn]);
            break;
            
        case 'permissions':
            $auth->requireAuth();
            
            $permissions = [];
            $allPermissions = [
                'view_agendamentos', 'create_agendamentos', 'update_agendamentos',
                'view_pacientes', 'create_pacientes', 'update_pacientes',
                'view_fila', 'manage_fila', 'call_patient'
            ];
            
            foreach ($allPermissions as $permission) {
                $permissions[$permission] = $auth->hasPermission($permission);
            }
            
            jsonResponse(['success' => true, 'permissions' => $permissions]);
            break;
            
        case 'csrf-token':
            $token = gerarCsrfToken();
            jsonResponse(['success' => true, 'token' => $token]);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePost($action, $auth) {
    $data = getJsonInput();
    
    switch ($action) {
        case 'login':
            $email = $data['email'] ?? '';
            $senha = $data['senha'] ?? '';
            
            if (empty($email) || empty($senha)) {
                jsonResponse(['success' => false, 'message' => 'Email e senha são obrigatórios'], 400);
            }
            
            $result = $auth->login($email, $senha);
            
            if ($result['success']) {
                jsonResponse($result);
            } else {
                jsonResponse($result, 401);
            }
            break;
            
        case 'logout':
            $result = $auth->logout();
            jsonResponse($result);
            break;
            
        case 'register':
            $auth->requirePermission('*'); // Apenas admin pode criar usuários
            
            $result = $auth->createUser($data);
            
            if ($result['success']) {
                jsonResponse($result, 201);
            } else {
                jsonResponse($result, 400);
            }
            break;
            
        case 'change-password':
            $auth->requireAuth();
            
            $userId = $_SESSION['user_id'];
            $senhaAtual = $data['senha_atual'] ?? '';
            $novaSenha = $data['nova_senha'] ?? '';
            $confirmarSenha = $data['confirmar_senha'] ?? '';
            
            if (empty($senhaAtual) || empty($novaSenha) || empty($confirmarSenha)) {
                jsonResponse(['success' => false, 'message' => 'Todos os campos são obrigatórios'], 400);
            }
            
            if ($novaSenha !== $confirmarSenha) {
                jsonResponse(['success' => false, 'message' => 'Nova senha e confirmação não coincidem'], 400);
            }
            
            if (strlen($novaSenha) < 6) {
                jsonResponse(['success' => false, 'message' => 'Nova senha deve ter pelo menos 6 caracteres'], 400);
            }
            
            $result = $auth->changePassword($userId, $senhaAtual, $novaSenha);
            
            if ($result['success']) {
                jsonResponse($result);
            } else {
                jsonResponse($result, 400);
            }
            break;
            
        case 'forgot-password':
            $email = $data['email'] ?? '';
            
            if (empty($email)) {
                jsonResponse(['success' => false, 'message' => 'Email é obrigatório'], 400);
            }
            
            $result = $auth->requestPasswordReset($email);
            jsonResponse($result);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePut($action, $auth) {
    $data = getJsonInput();
    
    switch ($action) {
        case 'update-profile':
            $auth->requireAuth();
            
            $userId = $_SESSION['user_id'];
            $allowedFields = ['nome', 'email', 'telefone'];
            
            $updateData = [];
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $updateData[$field] = $data[$field];
                }
            }
            
            if (empty($updateData)) {
                jsonResponse(['success' => false, 'message' => 'Nenhum campo válido para atualizar'], 400);
            }
            
            // Validar email se fornecido
            if (isset($updateData['email']) && !validarEmail($updateData['email'])) {
                jsonResponse(['success' => false, 'message' => 'Email inválido'], 400);
            }
            
            try {
                $db = Database::getInstance();
                $result = $db->update('usuarios', $updateData, 'id = :id', ['id' => $userId]);
                
                if ($result) {
                    jsonResponse(['success' => true, 'message' => 'Perfil atualizado com sucesso']);
                } else {
                    jsonResponse(['success' => false, 'message' => 'Erro ao atualizar perfil'], 400);
                }
            } catch (Exception $e) {
                jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}
?>

