<?php
/**
 * CliniSys - API de Agendamentos
 * Endpoints para gerenciar agendamentos
 */

require_once '../includes/autoload.php';

header('Content-Type: application/json; charset=utf-8');

$auth = new Auth();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGet($path, $auth);
            break;
        case 'POST':
            handlePost($path, $auth);
            break;
        case 'PUT':
            handlePut($path, $auth);
            break;
        case 'DELETE':
            handleDelete($path, $auth);
            break;
        default:
            jsonResponse(['success' => false, 'message' => 'Método não permitido'], 405);
    }
} catch (Exception $e) {
    logError('Erro na API de agendamentos: ' . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Erro interno do servidor'], 500);
}

function handleGet($action, $auth) {
    $agendamento = new Agendamento();
    
    switch ($action) {
        case 'list':
            $auth->requirePermission('view_agendamentos');
            
            $filters = [
                'data_inicio' => $_GET['data_inicio'] ?? null,
                'data_fim' => $_GET['data_fim'] ?? null,
                'medico_id' => $_GET['medico_id'] ?? null,
                'paciente_id' => $_GET['paciente_id'] ?? null,
                'status' => $_GET['status'] ?? null
            ];
            
            $agendamentos = $agendamento->getAllWithDetails($filters);
            jsonResponse(['success' => true, 'data' => $agendamentos]);
            break;
            
        case 'calendar':
            $auth->requirePermission('view_agendamentos');
            
            $start = $_GET['start'] ?? date('Y-m-d');
            $end = $_GET['end'] ?? date('Y-m-d', strtotime('+1 month'));
            $medicoId = $_GET['medico_id'] ?? null;
            
            $events = $agendamento->getForCalendar($start, $end, $medicoId);
            jsonResponse($events);
            break;
            
        case 'show':
            $auth->requirePermission('view_agendamentos');
            
            $id = $_GET['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $dados = $agendamento->find($id);
            if (!$dados) {
                jsonResponse(['success' => false, 'message' => 'Agendamento não encontrado'], 404);
            }
            
            jsonResponse(['success' => true, 'data' => $dados]);
            break;
            
        case 'today':
            $auth->requirePermission('view_agendamentos');
            
            $data = $_GET['data'] ?? date('Y-m-d');
            $medicoId = $_GET['medico_id'] ?? null;
            
            $agendamentos = $agendamento->getAgendamentosDoDia($data, $medicoId);
            jsonResponse(['success' => true, 'data' => $agendamentos]);
            break;
            
        case 'stats':
            $auth->requirePermission('view_agendamentos');
            
            $periodo = $_GET['periodo'] ?? '30 days';
            $medicoId = $_GET['medico_id'] ?? null;
            
            $stats = $agendamento->getEstatisticas($periodo, $medicoId);
            jsonResponse(['success' => true, 'data' => $stats]);
            break;
            
        case 'available-times':
            $auth->requirePermission('view_agendamentos');
            
            $medicoId = $_GET['medico_id'] ?? null;
            $data = $_GET['data'] ?? null;
            $duracao = $_GET['duracao'] ?? 30;
            
            if (!$medicoId || !$data) {
                jsonResponse(['success' => false, 'message' => 'Médico e data são obrigatórios'], 400);
            }
            
            $medico = new Medico();
            $horarios = $medico->getHorariosDisponiveis($medicoId, $data, $duracao);
            
            jsonResponse(['success' => true, 'data' => $horarios]);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePost($action, $auth) {
    $agendamento = new Agendamento();
    $data = getJsonInput();
    
    switch ($action) {
        case 'create':
            $auth->requirePermission('create_agendamentos');
            
            $result = $agendamento->createAgendamento($data);
            
            if ($result['success']) {
                jsonResponse($result, 201);
            } else {
                jsonResponse($result, 400);
            }
            break;
            
        case 'confirm':
            $auth->requirePermission('update_agendamentos');
            
            $id = $data['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $agendamento->confirmar($id);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Agendamento confirmado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao confirmar agendamento'], 400);
            }
            break;
            
        case 'start':
            $auth->requirePermission('update_agendamentos');
            
            $id = $data['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $agendamento->iniciarAtendimento($id);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Atendimento iniciado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao iniciar atendimento'], 400);
            }
            break;
            
        case 'finish':
            $auth->requirePermission('update_agendamentos');
            
            $id = $data['id'] ?? null;
            $observacoes = $data['observacoes'] ?? null;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $agendamento->finalizarAtendimento($id, $observacoes);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Atendimento finalizado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao finalizar atendimento'], 400);
            }
            break;
            
        case 'reschedule':
            $auth->requirePermission('update_agendamentos');
            
            $id = $data['id'] ?? null;
            $novaDataHora = $data['nova_data_hora'] ?? null;
            
            if (!$id || !$novaDataHora) {
                jsonResponse(['success' => false, 'message' => 'ID e nova data/hora são obrigatórios'], 400);
            }
            
            $result = $agendamento->reagendar($id, $novaDataHora);
            jsonResponse($result);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handlePut($action, $auth) {
    $agendamento = new Agendamento();
    $data = getJsonInput();
    
    switch ($action) {
        case 'update':
            $auth->requirePermission('update_agendamentos');
            
            $id = $data['id'] ?? null;
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            unset($data['id']); // Remover ID dos dados de atualização
            
            $errors = $agendamento->validateAgendamento($data);
            if (!empty($errors)) {
                jsonResponse(['success' => false, 'errors' => $errors], 400);
            }
            
            $result = $agendamento->update($id, $data);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Agendamento atualizado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao atualizar agendamento'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}

function handleDelete($action, $auth) {
    $agendamento = new Agendamento();
    
    switch ($action) {
        case 'cancel':
            $auth->requirePermission('update_agendamentos');
            
            $id = $_GET['id'] ?? null;
            $motivo = $_GET['motivo'] ?? 'Cancelado pelo sistema';
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $agendamento->cancelar($id, $motivo);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Agendamento cancelado']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao cancelar agendamento'], 400);
            }
            break;
            
        case 'no-show':
            $auth->requirePermission('update_agendamentos');
            
            $id = $_GET['id'] ?? null;
            
            if (!$id) {
                jsonResponse(['success' => false, 'message' => 'ID obrigatório'], 400);
            }
            
            $result = $agendamento->marcarFalta($id);
            
            if ($result) {
                jsonResponse(['success' => true, 'message' => 'Falta registrada']);
            } else {
                jsonResponse(['success' => false, 'message' => 'Erro ao registrar falta'], 400);
            }
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Ação não encontrada'], 404);
    }
}
?>

