<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header("Location: index.php");
    exit();
}
require_once '../conexao.php';

// Função para processar upload de foto
function processarUploadFoto($arquivo) {
    if (empty($arquivo['tmp_name'])) {
        return null; // Nenhum arquivo enviado
    }
    
    $extensoes_permitidas = ['jpg', 'jpeg', 'png', 'gif'];
    $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));
    
    if (!in_array($extensao, $extensoes_permitidas)) {
        throw new Exception("Formato de arquivo não permitido. Use JPG, PNG ou GIF.");
    }
    
    if ($arquivo['size'] > 5 * 1024 * 1024) { // 5MB
        throw new Exception("Arquivo muito grande. Tamanho máximo: 5MB.");
    }
    
    // Criar nome único para o arquivo
    $nome_arquivo = 'medico_' . uniqid() . '.jpg';
    $caminho_completo = '../upload/' . $nome_arquivo;
    
    // Verificar se o arquivo existe e é legível
    if (!file_exists($arquivo['tmp_name']) || !is_readable($arquivo['tmp_name'])) {
        throw new Exception("Arquivo temporário não encontrado ou não é legível.");
    }
    
    // Detectar o tipo real da imagem
    $info = @getimagesize($arquivo['tmp_name']);
    if ($info === false) {
        throw new Exception("Arquivo não é uma imagem válida.");
    }
    
    // Usar o tipo real da imagem, não a extensão do arquivo
    $tipo_real = $info[2];
    $imagem = false;
    
    switch ($tipo_real) {
        case IMAGETYPE_JPEG:
            $imagem = @imagecreatefromjpeg($arquivo['tmp_name']);
            break;
        case IMAGETYPE_PNG:
            $imagem = @imagecreatefrompng($arquivo['tmp_name']);
            break;
        case IMAGETYPE_GIF:
            $imagem = @imagecreatefromgif($arquivo['tmp_name']);
            break;
        default:
            throw new Exception("Formato de imagem não suportado: " . image_type_to_extension($tipo_real));
    }
    
    if (!$imagem) {
        throw new Exception("Erro ao processar a imagem.");
    }
    
    // Obter dimensões originais
    $largura_original = imagesx($imagem);
    $altura_original = imagesy($imagem);
    
    // Calcular novas dimensões (máximo 500x500)
    $max_tamanho = 500;
    if ($largura_original > $altura_original) {
        $nova_largura = min($max_tamanho, $largura_original);
        $nova_altura = ($altura_original * $nova_largura) / $largura_original;
    } else {
        $nova_altura = min($max_tamanho, $altura_original);
        $nova_largura = ($largura_original * $nova_altura) / $altura_original;
    }
    
    // Criar nova imagem redimensionada com fundo branco
    $nova_imagem = imagecreatetruecolor($nova_largura, $nova_altura);
    $fundo_branco = imagecolorallocate($nova_imagem, 255, 255, 255);
    imagefill($nova_imagem, 0, 0, $fundo_branco);
    
    imagecopyresampled($nova_imagem, $imagem, 0, 0, 0, 0, $nova_largura, $nova_altura, $largura_original, $altura_original);
    
    // Salvar como JPG com qualidade 80%
    if (!imagejpeg($nova_imagem, $caminho_completo, 80)) {
        throw new Exception("Erro ao salvar a imagem.");
    }
    
    // Limpar memória
    imagedestroy($imagem);
    imagedestroy($nova_imagem);
    
    return 'upload/' . $nome_arquivo;
}

$acao = $_POST['acao'] ?? $_GET['acao'] ?? '';

switch ($acao) {
    case 'adicionar':
        if (empty($_POST['nome_completo']) || empty($_POST['whatsapp']) || empty($_POST['senha']) || empty($_POST['especialidades'])) {
            $_SESSION['error_message'] = "Erro: Nome, WhatsApp, Senha e ao menos uma Especialidade são obrigatórios.";
            header("Location: medicos_adicionar.php");
            exit();
        }

        // Verificar se WhatsApp já existe
        $whatsapp_check = $conexao->prepare("SELECT id FROM medicos WHERE whatsapp = ?");
        $whatsapp_check->bind_param("s", $_POST['whatsapp']);
        $whatsapp_check->execute();
        $result = $whatsapp_check->get_result();
        if ($result->num_rows > 0) {
            $_SESSION['error_message'] = "Erro: Este número de WhatsApp já está cadastrado para outro médico.";
            $whatsapp_check->close();
            header("Location: medicos_adicionar.php");
            exit();
        }
        $whatsapp_check->close();

        $conexao->begin_transaction();
        try {
            // Processar upload da foto
            $caminho_foto = null;
            if (isset($_FILES['foto']) && !empty($_FILES['foto']['tmp_name']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
                $caminho_foto = processarUploadFoto($_FILES['foto']);
            }
            
            $sql_medico = "INSERT INTO medicos (nome_completo, endereco, whatsapp, email, foto, senha, duracao_consulta_min) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt_medico = $conexao->prepare($sql_medico);
            if ($stmt_medico === false) throw new Exception("Erro ao preparar query do médico: " . $conexao->error);

            $senha_hash = password_hash($_POST['senha'], PASSWORD_DEFAULT);
            $duracao = (int)($_POST['duracao_consulta_min'] ?? 30);
            $email = empty(trim($_POST['email'])) ? null : trim($_POST['email']);
            $endereco = empty(trim($_POST['endereco'])) ? null : trim($_POST['endereco']);

            $stmt_medico->bind_param("ssssssi", $_POST['nome_completo'], $endereco, $_POST['whatsapp'], $email, $caminho_foto, $senha_hash, $duracao);
            
            if (!$stmt_medico->execute()) throw new Exception("Erro ao inserir médico: " . $stmt_medico->error);
            
            $id_medico = $conexao->insert_id;
            if ($id_medico === 0) throw new Exception("Não foi possível obter o ID do novo médico.");
            
            $stmt_medico->close();

            $sql_espec = "INSERT INTO medico_especialidades (id_medico, id_especialidade) VALUES (?, ?)";
            $stmt_espec = $conexao->prepare($sql_espec);
            if ($stmt_espec === false) throw new Exception("Erro ao preparar query das especialidades: " . $conexao->error);

            foreach ($_POST['especialidades'] as $id_especialidade) {
                $stmt_espec->bind_param("ii", $id_medico, $id_especialidade);
                if (!$stmt_espec->execute()) throw new Exception("Erro ao vincular especialidade: " . $stmt_espec->error);
            }
            $stmt_espec->close();
            
            $conexao->commit();
            $_SESSION['success_message'] = "Médico adicionado com sucesso!";

        } catch (Exception $e) {
            $conexao->rollback();
            $_SESSION['error_message'] = $e->getMessage();
        }
        break;

    case 'editar':
        $id = $_POST['id'];
        if (empty($_POST['nome_completo']) || empty($_POST['whatsapp']) || empty($_POST['especialidades'])) {
            $_SESSION['error_message'] = "Erro: Nome, WhatsApp e ao menos uma Especialidade são obrigatórios.";
            header("Location: medicos_editar.php?id=$id");
            exit();
        }

        // Verificar se WhatsApp já existe para outro médico
        $whatsapp_check = $conexao->prepare("SELECT id FROM medicos WHERE whatsapp = ? AND id != ?");
        $whatsapp_check->bind_param("si", $_POST['whatsapp'], $id);
        $whatsapp_check->execute();
        $result = $whatsapp_check->get_result();
        if ($result->num_rows > 0) {
            $_SESSION['error_message'] = "Erro: Este número de WhatsApp já está cadastrado para outro médico.";
            $whatsapp_check->close();
            header("Location: medicos_editar.php?id=$id");
            exit();
        }
        $whatsapp_check->close();

        $conexao->begin_transaction();
        try {
            // Processar upload da foto se houver
            $caminho_foto = null;
            if (isset($_FILES['foto']) && !empty($_FILES['foto']['tmp_name']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
                $caminho_foto = processarUploadFoto($_FILES['foto']);
            }
            
            // Atualiza dados do médico
            $sql = "UPDATE medicos SET nome_completo=?, endereco=?, whatsapp=?, email=?, duracao_consulta_min=?";
            $endereco = empty(trim($_POST['endereco'])) ? null : trim($_POST['endereco']);
            $email = empty(trim($_POST['email'])) ? null : trim($_POST['email']);
            $params = [$_POST['nome_completo'], $endereco, $_POST['whatsapp'], $email, (int)$_POST['duracao_consulta_min']];
            $types = "ssssi";

            // Atualiza foto se foi enviada
            if ($caminho_foto) {
                $sql .= ", foto=?";
                $params[] = $caminho_foto;
                $types .= "s";
            }

            // Atualiza senha se foi informada
            if (!empty($_POST['senha'])) {
                $sql .= ", senha=?";
                $params[] = password_hash($_POST['senha'], PASSWORD_DEFAULT);
                $types .= "s";
            }
            $sql .= " WHERE id=?";
            $params[] = $id;
            $types .= "i";

            $stmt = $conexao->prepare($sql);
            $stmt->bind_param($types, ...$params);
            if (!$stmt->execute()) throw new Exception("Erro ao atualizar médico: " . $stmt->error);
            $stmt->close();

            // Atualiza especialidades
            $conexao->query("DELETE FROM medico_especialidades WHERE id_medico = $id");
            $sql_espec = "INSERT INTO medico_especialidades (id_medico, id_especialidade) VALUES (?, ?)";
            $stmt_espec = $conexao->prepare($sql_espec);
            foreach ($_POST['especialidades'] as $id_especialidade) {
                $stmt_espec->bind_param("ii", $id, $id_especialidade);
                if (!$stmt_espec->execute()) throw new Exception("Erro ao vincular especialidade: " . $stmt_espec->error);
            }
            $stmt_espec->close();

            $conexao->commit();
            $_SESSION['success_message'] = "Médico atualizado com sucesso!";
        } catch (Exception $e) {
            $conexao->rollback();
            $_SESSION['error_message'] = $e->getMessage();
        }
        break;

    case 'toggle_status':
        $id = $_GET['id'];
        // Busca o status atual
        $result = $conexao->query("SELECT ativo FROM medicos WHERE id = $id");
        if ($result && $row = $result->fetch_assoc()) {
            $novo_status = $row['ativo'] ? 0 : 1;
            $conexao->query("UPDATE medicos SET ativo = $novo_status WHERE id = $id");
            $_SESSION['success_message'] = $novo_status ? "Médico ativado com sucesso!" : "Médico desativado com sucesso!";
        }
        break;

    case 'excluir':
        $id = $_GET['id'];
        
        $conexao->begin_transaction();
        try {
            // Verificar se o médico tem agendamentos
            $check_agendamentos = $conexao->query("SELECT COUNT(*) as total FROM agendamentos WHERE id_medico = $id");
            $agendamentos = $check_agendamentos->fetch_assoc();
            
            if ($agendamentos['total'] > 0) {
                throw new Exception("Não é possível excluir este médico pois ele possui agendamentos cadastrados.");
            }
            
            // Excluir especialidades do médico
            $conexao->query("DELETE FROM medico_especialidades WHERE id_medico = $id");
            
            // Buscar e excluir foto se existir
            $foto_result = $conexao->query("SELECT foto FROM medicos WHERE id = $id");
            if ($foto_result && $foto_row = $foto_result->fetch_assoc()) {
                if (!empty($foto_row['foto']) && file_exists('../' . $foto_row['foto'])) {
                    unlink('../' . $foto_row['foto']);
                }
            }
            
            // Excluir médico
            $conexao->query("DELETE FROM medicos WHERE id = $id");
            
            $conexao->commit();
            $_SESSION['success_message'] = "Médico excluído com sucesso!";
            
        } catch (Exception $e) {
            $conexao->rollback();
            $_SESSION['error_message'] = $e->getMessage();
        }
        break;
}

$conexao->close();
header("Location: medicos_listar.php");
exit();
?>