<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="refresh" content="300">
    <title>Painel de Chamadas - CliniSys</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        :root {
            --primary-color: #1a56db;
            --secondary-color: #7c3aed;
            --accent-color: #f59e0b;
            --text-light: #f3f4f6;
            --text-dark: #1f2937;
            --bg-dark: #111827;
            --bg-card: #1e293b;
        }

        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');

        body {
            background-color: var(--bg-dark);
            color: var(--text-light);
            font-family: 'Inter', sans-serif;
            overflow: hidden;
            margin: 0;
            padding: 0;
            height: 100vh;
        }

        .container-fluid {
            display: flex;
            height: 100vh;
            padding: 0;
        }

        .main-call {
            flex-grow: 1;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            padding: 2rem;
            position: relative;
            overflow: hidden;
        }

        .main-call::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg width="20" height="20" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><rect width="1" height="1" fill="rgba(255,255,255,0.1)"/></svg>');
            opacity: 0.1;
        }

        .main-call.blink {
            animation: blink-animation 1s ease;
        }

        @keyframes blink-animation {
            0% { transform: scale(1); }
            50% { transform: scale(1.02); }
            100% { transform: scale(1); }
        }

        .main-call .patient-name {
            font-size: 8vw;
            font-weight: 700;
            text-transform: uppercase;
            line-height: 1.1;
            text-align: center;
            margin: 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
            position: relative;
            z-index: 1;
        }

        .main-call .consultorio {
            font-size: 5vw;
            font-weight: 600;
            margin-top: 1rem;
            background: rgba(255,255,255,0.1);
            padding: 0.5rem 2rem;
            border-radius: 9999px;
            backdrop-filter: blur(10px);
            position: relative;
            z-index: 1;
        }

        .last-calls {
            width: 35%;
            background-color: var(--bg-card);
            padding: 2rem;
            display: flex;
            flex-direction: column;
            box-shadow: -4px 0 15px rgba(0,0,0,0.1);
        }

        .last-calls h2 {
            font-weight: 700;
            color: var(--accent-color);
            text-align: center;
            margin-bottom: 2rem;
            font-size: 1.5rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .last-calls .call-item {
            background: rgba(255,255,255,0.05);
            border-radius: 1rem;
            padding: 1rem;
            margin-bottom: 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
            border: 1px solid rgba(255,255,255,0.1);
        }

        .last-calls .call-item:hover {
            transform: translateX(-5px);
            background: rgba(255,255,255,0.1);
        }

        .last-calls .patient-name-small {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-light);
        }

        .last-calls .consultorio-small {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--accent-color);
            background: rgba(245,158,11,0.1);
            padding: 0.25rem 1rem;
            border-radius: 9999px;
        }

        #clock {
            font-size: 2rem;
            font-weight: 700;
            text-align: center;
            margin-top: auto;
            padding: 1rem;
            background: rgba(255,255,255,0.05);
            border-radius: 1rem;
            border: 1px solid rgba(255,255,255,0.1);
        }

        .empty-state {
            text-align: center;
            padding: 2rem;
            color: var(--text-light);
            opacity: 0.7;
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: var(--accent-color);
        }

        @media (max-width: 768px) {
            .container-fluid {
                flex-direction: column;
            }

            .main-call {
                height: 60vh;
            }

            .last-calls {
                width: 100%;
                height: 40vh;
                padding: 1rem;
            }

            .main-call .patient-name {
                font-size: 12vw;
            }

            .main-call .consultorio {
                font-size: 8vw;
            }

            .last-calls .patient-name-small,
            .last-calls .consultorio-small {
                font-size: 1rem;
            }

            #clock {
                font-size: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div id="main-call-panel" class="main-call">
            <div id="main-patient-name" class="patient-name">Aguardando...</div>
            <div id="main-consultorio" class="consultorio"></div>
        </div>
        <div class="last-calls">
            <h2>Últimas Chamadas</h2>
            <div id="last-calls-list"></div>
            <div id="clock"></div>
        </div>
    </div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const mainCallPanel = document.getElementById('main-call-panel');
    const mainPatientNameEl = document.getElementById('main-patient-name');
    const mainConsultorioEl = document.getElementById('main-consultorio');
    const lastCallsListEl = document.getElementById('last-calls-list');
    const clockEl = document.getElementById('clock');
    
    let ultimoIdAnunciado = null;
    let ultimaChamadaVoz = null;

    function falarTexto(texto) {
        if ('speechSynthesis' in window) {
            // Cancela qualquer fala em andamento
            window.speechSynthesis.cancel();
            
            const utterance = new SpeechSynthesisUtterance(texto);
            utterance.lang = 'pt-BR';
            utterance.rate = 0.9;
            utterance.volume = 1;
            
            // Armazena a chamada atual
            ultimaChamadaVoz = utterance;
            
            window.speechSynthesis.speak(utterance);
        }
    }

    function atualizarPainel() {
        fetch('api/get_chamadas_tv.php')
        .then(response => response.json())
        .then(data => {
            if (data.length > 0) {
                const chamadaMaisRecente = data[0];
                
                // Atualiza o painel principal
                mainPatientNameEl.textContent = chamadaMaisRecente.nome_paciente;
                mainConsultorioEl.textContent = `${chamadaMaisRecente.numero_consultorio}`;

                // Verifica se é uma nova chamada
                if (chamadaMaisRecente.id !== ultimoIdAnunciado) {
                    ultimoIdAnunciado = chamadaMaisRecente.id;
                    
                    // Efeito visual
                    mainCallPanel.classList.add('blink');
                    setTimeout(() => mainCallPanel.classList.remove('blink'), 1000);
                    
                    // Anúncio por voz
                    const textoParaFalar = `${chamadaMaisRecente.nome_paciente}, por favor, dirija-se ao ${chamadaMaisRecente.numero_consultorio}.`;
                    falarTexto(textoParaFalar);
                }

                // Atualiza a lista de últimas chamadas
                lastCallsListEl.innerHTML = '';
                for (let i = 1; i < data.length; i++) {
                    const chamada = data[i];
                    const item = `
                        <div class="call-item">
                            <span class="patient-name-small">${chamada.nome_paciente}</span>
                            <span class="consultorio-small">${chamada.numero_consultorio}</span>
                        </div>
                    `;
                    lastCallsListEl.innerHTML += item;
                }
            } else {
                mainPatientNameEl.textContent = 'Aguardando...';
                mainConsultorioEl.textContent = '';
                lastCallsListEl.innerHTML = `
                    <div class="empty-state">
                        <i class="bi bi-emoji-smile"></i>
                        <p>Nenhuma chamada no momento</p>
                    </div>`;
                ultimoIdAnunciado = null;
            }
        })
        .catch(error => {
            console.error('Erro ao atualizar painel:', error);
        });
    }
    
    function atualizarRelogio() {
        const agora = new Date();
        const horas = String(agora.getHours()).padStart(2, '0');
        const minutos = String(agora.getMinutes()).padStart(2, '0');
        const segundos = String(agora.getSeconds()).padStart(2, '0');
        clockEl.textContent = `${horas}:${minutos}:${segundos}`;
    }

    // Inicialização
    atualizarPainel();
    atualizarRelogio();
    
    // Atualizações periódicas
    setInterval(atualizarPainel, 5000);
    setInterval(atualizarRelogio, 1000);
});
</script>
</body>
</html>