<?php
header('Content-Type: application/json');
require_once '../conexao.php';

$dados = json_decode(file_get_contents('php://input'), true);

$idMedico = (int)($dados['medicoId'] ?? 0);
$dataHoraAgendamento = $dados['start'] ?? '';
$idEspecialidade = (int)($dados['especialidadeId'] ?? 0);
$idPaciente = (int)($dados['pacienteId'] ?? 0); // Para agendamentos internos
$nomePacienteDigitado = trim($dados['nome'] ?? '');
$telefonePaciente = trim($dados['telefone'] ?? '');

$conexao->begin_transaction();

try {
    $nomePacienteFinal = '';

    if ($idPaciente > 0) {
        // Fluxo Interno (Admin): O ID do paciente já foi fornecido.
        $stmt_find = $conexao->prepare("SELECT nome_completo FROM pacientes WHERE id = ?");
        $stmt_find->bind_param("i", $idPaciente);
        $stmt_find->execute();
        $paciente_existente = $stmt_find->get_result()->fetch_assoc();
        if (!$paciente_existente) {
            throw new Exception("Paciente com o ID fornecido não encontrado.");
        }
        $nomePacienteFinal = $paciente_existente['nome_completo'];
        $stmt_find->close();
    } else {
        // Fluxo Público: Busca ou cria paciente pelo telefone
        if (empty($nomePacienteDigitado) || empty($telefonePaciente)) throw new Exception("Nome e telefone são obrigatórios.");
        
        $stmt_find = $conexao->prepare("SELECT id, nome_completo FROM pacientes WHERE telefone_contato = ?");
        $stmt_find->bind_param("s", $telefonePaciente);
        $stmt_find->execute();
        $resultado = $stmt_find->get_result();
        if ($resultado->num_rows > 0) {
            $paciente_existente = $resultado->fetch_assoc();
            $idPaciente = $paciente_existente['id'];
            $nomePacienteFinal = $paciente_existente['nome_completo'];
        } else {
            $stmt_insert = $conexao->prepare("INSERT INTO pacientes (nome_completo, telefone_contato) VALUES (?, ?)");
            $stmt_insert->bind_param("ss", $nomePacienteDigitado, $telefonePaciente);
            $stmt_insert->execute();
            $idPaciente = $conexao->insert_id;
            $nomePacienteFinal = $nomePacienteDigitado;
            $stmt_insert->close();
        }
        $stmt_find->close();
    }

    // Validações
    if (empty($idMedico) || empty($dataHoraAgendamento) || empty($idEspecialidade) || empty($idPaciente)) {
        throw new Exception("Dados essenciais para o agendamento estão faltando.");
    }
    
    // Verifica se slot já foi ocupado
    $stmt_check = $conexao->prepare("SELECT id FROM agendamentos WHERE id_medico = ? AND data_hora_agendamento = ?");
    $stmt_check->bind_param("is", $idMedico, $dataHoraAgendamento);
    $stmt_check->execute();
    if ($stmt_check->get_result()->num_rows > 0) {
        throw new Exception("Este horário acabou de ser preenchido. Por favor, escolha outro.");
    }
    $stmt_check->close();

    // Insere o agendamento
    $stmt_agendamento = $conexao->prepare("INSERT INTO agendamentos (id_paciente, id_medico, id_especialidade_agendada, data_hora_agendamento, status_agendamento) VALUES (?, ?, ?, ?, 'agendado')");
    $stmt_agendamento->bind_param("iiss", $idPaciente, $idMedico, $idEspecialidade, $dataHoraAgendamento);
    if(!$stmt_agendamento->execute()) {
        throw new Exception("Não foi possível registrar o agendamento.");
    }
    $stmt_agendamento->close();

    $conexao->commit();
    echo json_encode(['success' => true, 'message' => 'Agendamento para ' . htmlspecialchars($nomePacienteFinal) . ' realizado com sucesso!']);

} catch (Exception $e) {
    $conexao->rollback();
    echo json_encode(['success' => false, 'message' => 'Ocorreu um erro: ' . $e->getMessage()]);
}
$conexao->close();
?>