<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Verificar login e permissões de admin
if (!isLoggedIn() || $_SESSION['user']['type'] !== 'admin') {
    header("Location: ../index.php");
    exit();
}

// Funções de gerenciamento
function addItem($type, $data) {
    $db = Database::getInstance()->getConnection();
    
    switch ($type) {
        case 'office':
            $stmt = $db->prepare("INSERT INTO offices (id, name) VALUES (UUID(), :name)");
            return $stmt->execute([':name' => $data['name']]);
            
        case 'procedure':
            $stmt = $db->prepare("
                INSERT INTO procedures (id, name, value, commission) 
                VALUES (UUID(), :name, :value, :commission)
            ");
            return $stmt->execute([
                ':name' => $data['name'],
                ':value' => str_replace(['R$', '.', ','], ['', '', '.'], $data['value']),
                ':commission' => str_replace(['R$', '.', ','], ['', '', '.'], $data['commission'])
            ]);
            
        case 'specialty':
            $stmt = $db->prepare("INSERT INTO specialties (id, name) VALUES (UUID(), :name)");
            return $stmt->execute([':name' => $data['name']]);
            
        case 'payment':
            $stmt = $db->prepare("
                INSERT INTO payment_methods (id, name, discount, tax) 
                VALUES (UUID(), :name, :discount, :tax)
            ");
            return $stmt->execute([
                ':name' => $data['name'],
                ':discount' => str_replace(['R$', '.', ','], ['', '', '.'], $data['discount']),
                ':tax' => str_replace(['%', ','], ['', '.'], $data['tax'])
            ]);
            
        case 'attendance':
            $stmt = $db->prepare("INSERT INTO attendance_types (id, name) VALUES (UUID(), :name)");
            return $stmt->execute([':name' => $data['name']]);
    }
}

function editItem($type, $id, $data) {
    $db = Database::getInstance()->getConnection();
    
    switch ($type) {
        case 'office':
            $stmt = $db->prepare("UPDATE offices SET name = :name WHERE id = :id");
            return $stmt->execute([':id' => $id, ':name' => $data['name']]);
            
        case 'procedure':
            $stmt = $db->prepare("
                UPDATE procedures 
                SET name = :name, value = :value, commission = :commission 
                WHERE id = :id
            ");
            return $stmt->execute([
                ':id' => $id,
                ':name' => $data['name'],
                ':value' => str_replace(['R$', '.', ','], ['', '', '.'], $data['value']),
                ':commission' => str_replace(['R$', '.', ','], ['', '', '.'], $data['commission'])
            ]);
            
        case 'specialty':
            $stmt = $db->prepare("UPDATE specialties SET name = :name WHERE id = :id");
            return $stmt->execute([':id' => $id, ':name' => $data['name']]);
            
        case 'payment':
            $stmt = $db->prepare("
                UPDATE payment_methods 
                SET name = :name, discount = :discount, tax = :tax 
                WHERE id = :id
            ");
            return $stmt->execute([
                ':id' => $id,
                ':name' => $data['name'],
                ':discount' => str_replace(['R$', '.', ','], ['', '', '.'], $data['discount']),
                ':tax' => str_replace(['%', ','], ['', '.'], $data['tax'])
            ]);
            
        case 'attendance':
            $stmt = $db->prepare("UPDATE attendance_types SET name = :name WHERE id = :id");
            return $stmt->execute([':id' => $id, ':name' => $data['name']]);
    }
}

function deleteItem($type, $id) {
    $db = Database::getInstance()->getConnection();
    
    switch ($type) {
        case 'office':
            $stmt = $db->prepare("DELETE FROM offices WHERE id = :id");
            break;
        case 'procedure':
            $stmt = $db->prepare("DELETE FROM procedures WHERE id = :id");
            break;
        case 'specialty':
            $stmt = $db->prepare("DELETE FROM specialties WHERE id = :id");
            break;
        case 'payment':
            $stmt = $db->prepare("DELETE FROM payment_methods WHERE id = :id");
            break;
        case 'attendance':
            $stmt = $db->prepare("DELETE FROM attendance_types WHERE id = :id");
            break;
    }
    
    return $stmt->execute([':id' => $id]);
}

// Processar formulários
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = explode('_', $_POST['action']);
        $operation = $action[0];
        $type = $action[1];

        switch ($operation) {
            case 'add':
                addItem($type, $_POST);
                break;
            case 'edit':
                editItem($type, $_POST['id'], $_POST);
                break;
            case 'delete':
                deleteItem($type, $_POST['id']);
                break;
        }

        // Redirecionar após o processamento
        header("Location: " . $_SERVER['PHP_SELF'] . "?updated=true");
        exit();
    }
}

// Carregar dados do banco
$db = Database::getInstance()->getConnection();

// Carregar dados usando as funções do functions.php
$offices = getOffices();
$procedures = getProcedures();
$specialties = getSpecialties();
$attendance_types = getAttendanceTypes();
$payment_methods = getPaymentMethods();

// Definir seções
$sections = [
    ['title' => 'Consultórios', 'data' => $offices, 'type' => 'office'],
    ['title' => 'Procedimentos', 'data' => $procedures, 'type' => 'procedure'],
    ['title' => 'Especialidades', 'data' => $specialties, 'type' => 'specialty'],
    ['title' => 'Tipos de Atendimento', 'data' => $attendance_types, 'type' => 'attendance'],
    ['title' => 'Formas de Pagamento', 'data' => $payment_methods, 'type' => 'payment'],
];

?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Configurações</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8">
        <h1 class="text-3xl font-bold mb-6">Gerenciar Configurações</h1>

        <?php if (isset($_GET['updated']) && $_GET['updated'] === 'true'): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4" role="alert">
                <strong class="font-bold">Sucesso!</strong>
                <span class="block sm:inline"> As configurações foram atualizadas.</span>
            </div>
        <?php endif; ?>

        <div class="grid md:grid-cols-2 gap-6">
            <?php
            foreach ($sections as $section):
            ?>
            <div class="bg-white shadow rounded-lg p-6">
                <h2 class="text-2xl font-bold mb-4"><?= $section['title'] ?></h2>
                
                <!-- Formulário de Adição -->
                <div class="bg-gray-50 p-4 rounded mb-6">
                    <form method="post" class="space-y-4">
                    <input type="hidden" name="action" value="add_<?= $section['type'] ?>">
                        
                        <?php if ($section['type'] === 'procedure'): ?>
                            <div class="grid grid-cols-1 gap-4">
                                <div>
                                    <label class="block mb-2">Nome do Procedimento</label>
                                    <input type="text" name="name" 
                                        class="w-full border rounded py-2 px-3" required>
                                </div>
                                <div class="grid grid-cols-2 gap-4">
                                    <div>
                                        <label class="block mb-2">Valor</label>
                                        <input type="text" name="value" id="add_procedure_value"
                                            class="w-full border rounded py-2 px-3"
                                            placeholder="R$ 0,00" required>
                                    </div>
                                    <div>
                                        <label class="block mb-2">Comissão</label>
                                        <input type="text" name="commission" id="add_procedure_commission"
                                            class="w-full border rounded py-2 px-3"
                                            placeholder="R$ 0,00" required>
                                    </div>
                                </div>
                            </div>
                        <?php elseif ($section['type'] === 'payment'): ?>
                            <div class="grid grid-cols-1 gap-4">
                                <div>
                                    <label class="block mb-2">Nome da Forma de Pagamento</label>
                                    <input type="text" name="name" 
                                        class="w-full border rounded py-2 px-3" required>
                                </div>
                                <div class="grid grid-cols-2 gap-4">
                                    <div>
                                        <label class="block mb-2">Desconto</label>
                                        <input type="text" name="discount" id="add_payment_discount"
                                            class="w-full border rounded py-2 px-3"
                                            placeholder="R$ 0,00">
                                    </div>
                                    <div>
                                        <label class="block mb-2">Taxa (%)</label>
                                        <input type="text" name="tax" id="add_payment_tax"
                                            class="w-full border rounded py-2 px-3"
                                            placeholder="0,00%">
                                    </div>
                                </div>
                            </div>
                        <?php elseif ($section['type'] === 'specialty'): ?>
                            <div>
                                <label class="block mb-2">Nome</label>
                                <input type="text" name="name" 
                                    class="w-full border rounded py-2 px-3" required>
                            </div>
                        <?php elseif ($section['type'] === 'attendance'): ?>
                            <div>
                                <label class="block mb-2">Nome</label>
                                <input type="text" name="name" 
                                    class="w-full border rounded py-2 px-3" required>
                            </div>
                        <?php else: ?>
                            <div>
                                <label class="block mb-2">Nome</label>
                                <input type="text" name="name" 
                                    class="w-full border rounded py-2 px-3" required>
                            </div>
                        <?php endif; ?>
                        
                        <div class="flex justify-end">
                        <button type="submit" class="bg-green-500 hover:bg-green-600 text-white font-bold py-2 px-4 rounded">
                            Adicionar
                        </button>
                    </div>
                </form>
                </div>

                <!-- Lista de Itens -->
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="bg-gray-100">
                                <th class="px-4 py-2 text-left">Nome</th>
                                <?php if ($section['type'] === 'procedure'): ?>
                                    <th class="px-4 py-2 text-left">Valor</th>
                                    <th class="px-4 py-2 text-left">Comissão</th>
                                <?php elseif ($section['type'] === 'payment'): ?>
                                    <th class="px-4 py-2 text-left">Desconto</th>
                                    <th class="px-4 py-2 text-left">Taxa</th>
                                <?php endif; ?>
                                <th class="px-4 py-2 text-center">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                    <?php foreach ($section['data'] as $item): ?>
                                <tr class="border-t">
                                    <td class="px-4 py-2"><?= htmlspecialchars($item['name']) ?></td>
                                    <?php if ($section['type'] === 'procedure'): ?>
                                        <td class="px-4 py-2"><?= isset($item['value']) ? formatMoney($item['value']) : 'R$ 0,00' ?></td>
                                        <td class="px-4 py-2"><?= isset($item['commission']) ? formatMoney($item['commission']) : 'R$ 0,00' ?></td>
                                    <?php elseif ($section['type'] === 'payment'): ?>
                                        <td class="px-4 py-2"><?= isset($item['discount']) ? formatMoney($item['discount']) : 'R$ 0,00' ?></td>
                                        <td class="px-4 py-2"><?= isset($item['tax']) ? formatPercent($item['tax']) : '0,00%' ?></td>
                                    <?php endif; ?>
                                    <td class="px-4 py-2 text-center">
                                        <button onclick="openEditModal('<?= $section['type'] ?>', <?= htmlspecialchars(json_encode($item)) ?>)"
                                        class="bg-blue-500 hover:bg-blue-600 text-white font-bold py-1 px-3 rounded mr-2">
                                    Editar
                                </button>
                                        <button onclick="deleteItem('<?= $section['type'] ?>', '<?= $item['id'] ?>', '<?= htmlspecialchars($item['name']) ?>')"
                                                class="bg-red-500 hover:bg-red-600 text-white font-bold py-1 px-3 rounded">
                                        Excluir
                                    </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                            </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Modal de Edição -->
    <div id="editModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-bold mb-4">Editar Item</h3>
                <form id="editForm" method="post" class="space-y-4">
                    <input type="hidden" name="action" id="edit_action">
                    <input type="hidden" name="id" id="edit_id">
                    
                    <div id="editFields" class="space-y-4">
                        <!-- Campos serão inseridos dinamicamente via JavaScript -->
                    </div>

                    <div class="flex justify-end space-x-4">
                        <button type="button" onclick="closeEditModal()"
                                class="bg-gray-500 hover:bg-gray-600 text-white font-bold py-2 px-4 rounded">
                            Cancelar
                        </button>
                        <button type="submit"
                                class="bg-blue-500 hover:bg-blue-600 text-white font-bold py-2 px-4 rounded">
                            Salvar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Aplicar máscaras aos campos de adicionar
        document.getElementById('add_procedure_value')?.addEventListener('input', function() {
            maskMoney(this);
        });
        document.getElementById('add_procedure_commission')?.addEventListener('input', function() {
            maskMoney(this);
        });
        document.getElementById('add_payment_discount')?.addEventListener('input', function() {
            maskMoney(this);
        });
        document.getElementById('add_payment_tax')?.addEventListener('input', function() {
            maskPercentage(this);
        });

        // Função para máscara monetária
        function maskMoney(input) {
            let value = input.value.replace(/\D/g, '');
            value = (parseFloat(value) / 100).toFixed(2);
            value = value.replace('.', ',');
            value = value.replace(/(\d)(?=(\d{3})+(?!\d))/g, '$1.');
            input.value = 'R$ ' + value;
        }

        // Função para máscara de percentual
        function maskPercentage(input) {
            let value = input.value.replace(/\D/g, '');
            value = (parseFloat(value) / 100).toFixed(2);
            value = value.replace('.', ',');
            input.value = value + '%';
        }

        // Modal de edição
        const modal = document.getElementById('editModal');
        const editForm = document.getElementById('editForm');
        const editFields = document.getElementById('editFields');

        function openEditModal(type, item) {
            modal.classList.remove('hidden');
            
            document.getElementById('edit_action').value = 'edit_' + type;
            document.getElementById('edit_id').value = item.id;

            let fieldsHtml = `
                <div>
                    <label class="block mb-2">Nome</label>
                    <input type="text" name="name" class="w-full border rounded py-2 px-3" 
                        value="${item.name}" required>
                </div>
            `;

            // Adiciona campos específicos para cada tipo
            switch(type) {
                case 'procedure':
                    fieldsHtml += `
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block mb-2">Valor</label>
                                <input type="text" name="value" class="w-full border rounded py-2 px-3 money-mask"
                                    value="${item.value ? formatMoney(item.value) : 'R$ 0,00'}" required>
                            </div>
                            <div>
                                <label class="block mb-2">Comissão</label>
                                <input type="text" name="commission" class="w-full border rounded py-2 px-3 money-mask"
                                    value="${item.commission ? formatMoney(item.commission) : 'R$ 0,00'}" required>
                            </div>
                        </div>
                    `;
                    break;

                case 'payment':
                    fieldsHtml += `
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block mb-2">Desconto</label>
                                <input type="text" name="discount" class="w-full border rounded py-2 px-3 money-mask"
                                    value="${item.discount ? formatMoney(item.discount) : 'R$ 0,00'}">
                            </div>
                            <div>
                                <label class="block mb-2">Taxa (%)</label>
                                <input type="text" name="tax" class="w-full border rounded py-2 px-3 percentage-mask"
                                    value="${item.tax ? formatPercent(item.tax) : '0,00%'}">
                            </div>
                        </div>
                    `;
                    break;
            }

            editFields.innerHTML = fieldsHtml;

            // Aplicar máscaras aos campos monetários
            document.querySelectorAll('.money-mask').forEach(input => {
                input.addEventListener('input', function() {
                    maskMoney(this);
                });
                // Aplicar máscara inicial se o campo estiver vazio
                if (!input.value) {
                    maskMoney(input);
                }
            });

            // Aplicar máscaras aos campos de percentual
            document.querySelectorAll('.percentage-mask').forEach(input => {
                input.addEventListener('input', function() {
                    maskPercentage(this);
                });
                // Aplicar máscara inicial se o campo estiver vazio
                if (!input.value) {
                    maskPercentage(input);
                }
            });
        }

        function closeEditModal() {
            modal.classList.add('hidden');
        }

        function deleteItem(type, id, name) {
            if (confirm(`Tem certeza que deseja excluir "${name}"?`)) {
                const form = document.createElement('form');
                form.method = 'post';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_${type}">
                    <input type="hidden" name="id" value="${id}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        // Fechar modal ao clicar fora
        modal.addEventListener('click', function(e) {
            if (e.target === modal) {
                closeEditModal();
            }
        });

        // Funções de formatação para o JavaScript
        function formatMoney(value) {
            if (!value) return 'R$ 0,00';
            return 'R$ ' + parseFloat(value).toFixed(2).replace('.', ',').replace(/(\d)(?=(\d{3})+(?!\d))/g, '$1.');
        }

        function formatPercent(value) {
            if (!value) return '0,00%';
            return parseFloat(value).toFixed(2).replace('.', ',') + '%';
        }
    </script>
</body>
</html>